/*
 * (c) GAFit toolkit $Id: rangecf.c 510 2025-04-22 14:23:13Z ro $
 */
#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "../nullist/nllist.h"
#include "../rstrings/rstrings.h"
#include "../inputline/line.h"
#include "../numbers.h"
#include "rangecf.h"




//cached results
char **AllStringTokens = NULL;
char **AllTemplateTokens = NULL;
char **AllWhiteSpaceTokens = NULL;

char *lparen[] = { LPAREN, NULL };
char *rparen[] = { RPAREN, NULL };
char *comma[] = { COMMA, NULL };
char *slash[] = { SLASH, NULL };
char *scolon[] = { SCOLON, NULL };
char *white[] = { SPACE, TAB, NULL };
char *dot[] = { DOT, NULL };
char *vbar[] = { VBAR, NULL };
char *colon[] = { COLON, NULL };
char *asterisk[] = { ASTERISK, NULL };
char *atsign[] = { AT, NULL };
char *bang[] = { BANG, NULL };
char *nsign[] = { NSIGN, NULL };
char *format[] = { NSIGN, DOT, NULL };

RANGECOEFS *
rcFactory (void)
{

  RANGECOEFS *nrc = malloc (sizeof (RANGECOEFS));
  nrc->strings = NULL;
  nrc->names = NULL;
  nrc->format = NULL;
  nrc->types = NULL;
  nrc->limits = NULL;
  nrc->llimit = NULL;
  nrc->ulimit = NULL;
  nrc->choice = NULL;
  nrc->decimals = NULL;
  nrc->coefs = NULL;
  nrc->ncoefs = 0;

  return nrc;
}

void
rcDelete (RANGECOEFS * rc)
{
  if (rc)
    {
      nllClear (rc->strings);
      nllClear (rc->names);
      nllClear (rc->format);
      nllClear (rc->types);
      nllClear (rc->limits);
      nllClear (rc->llimit);
      nllClear (rc->ulimit);
      nllClear (rc->choice);
      nllClear (rc->decimals);
      nllClear (rc->coefs);

      free (rc);
    }
}

void
rcPrint (RANGECOEFS * rc)
{
  nllPrint (rc->strings, "\nstrings\n");
  nllPrint (rc->names, "\nnames\n");
  nllPrint (rc->format, "\nformat\n");
  nllPrint (rc->types, "\ntypes\n");
  nllPrint (rc->limits, "\nlimits\n");
  nllPrint (rc->llimit, "\nllimit\n");
  nllPrint (rc->ulimit, "\nulimit\n");
  nllPrint (rc->choice, "\nchoice\n");
  nllPrint (rc->decimals, "\ndecimals\n");
  nllPrint (rc->coefs, "\ncoefficients\n");
}

/**
 * tokens to strings
 * */
char **
StringTokens (void)
{
  char **s = NULL;

  if (AllStringTokens)
    return AllStringTokens;

  s = nllBuild (LPAREN, RPAREN, COMMA, SCOLON, SPACE, TAB, SLASH, NULL);

  AllStringTokens = nllLenOrder (s);
  return AllStringTokens;
}

/**
 * tokens to templates
 * */
char **
TemplateTokens (void)
{
  char **s = NULL;

  if (AllTemplateTokens)
    return AllTemplateTokens;

  s =
    nllBuild (AT, LPAREN, RPAREN, COMMA, SCOLON, BANG, SPACE, TAB, NSIGN,
	      SLASH, NULL);

  AllTemplateTokens = nllLenOrder (s);

  return AllTemplateTokens;
}

/**
 * tokens to white space
 * */
char **
WhiteSpaceTokens (void)
{
  char **s = NULL;
  if (AllWhiteSpaceTokens)
    return AllWhiteSpaceTokens;
  s = nllBuild (SPACE, TAB, NULL);
  AllWhiteSpaceTokens = nllLenOrder (s);
  return AllWhiteSpaceTokens;
}

int
rcWriteAnalysisResults (RANGECOEFS * rc, char *outFile)
{
  FILE *w;
  char **s, **n, **t, **l, **i, **u, **c, **f, **d, **r;
  w = fopen (outFile, "w");
  if (w != NULL)
    {
      n = rc->names;
      s = rc->strings;
      f = rc->format;
      t = rc->types;
      l = rc->limits;
      i = rc->llimit;
      u = rc->ulimit;
      c = rc->choice;
      d = rc->decimals;
      r = rc->coefs;

      while (*s)
	{
	  fprintf (w, "%s|%s|%s|%s|%s|%s|%s|%s|%s|%s\n", *n, *s, *f, *t, *l,
		   *i, *u, *c, *d, *r);
	  n++;
	  s++;
	  f++;
	  t++;
	  l++;
	  i++;
	  u++;
	  c++;
	  d++;
	  r++;
	}
      fclose (w);
      return 1;
    }
  printf ("Error writing file: %s\n", outFile);
  return 0;
}

void
rcTemplateClean (RANGECOEFS * rc)
{
  rcDelete (rc);
  nllClear (AllTemplateTokens);
}

void
rcStringClean (RANGECOEFS * rc)
{
  rcDelete (rc);
  nllClear (AllStringTokens);
}


char **
rcAddOne (char **from, char **to, int n)
{
  return nllAddDup (to, nllString (from, n));
}

void
rcAddList (RANGECOEFS * rc, char **list)
{
  int num = 0;
  rc->names = rcAddOne (list, rc->names, num++);
  rc->strings = rcAddOne (list, rc->strings, num++);
  rc->format = rcAddOne (list, rc->format, num++);
  rc->types = rcAddOne (list, rc->types, num++);
  rc->limits = rcAddOne (list, rc->limits, num++);
  rc->llimit = rcAddOne (list, rc->llimit, num++);
  rc->ulimit = rcAddOne (list, rc->ulimit, num++);
  rc->choice = rcAddOne (list, rc->choice, num++);
  rc->decimals = rcAddOne (list, rc->decimals, num++);
  rc->coefs = rcAddOne (list, rc->coefs, num++);
}

void
rcFillOrder (RANGECOEFS * rc)
{
  for (int i = 0; i < rc->ncoefs; i++)
    rc->coefs = nllAdd (rc->coefs, sInt2Ascii (i + 1));
}

//---------------------------------------------
/**
 * analyzes a string containing a list of coef's specs 
 * without formats to build a rangeCoefs def.
 * (eg: used in mvariable)
 * */
RANGECOEFS *
rcAnalyzeString (char *s)
{
  char **list, **tmp, **p, **t;
  char *cp, *cc;
  RANGECOEFS *rc;
  int prev;

  rc = rcFactory ();

  list = nllParser (s, StringTokens);
  list = nllDelete (list, white);

  p = list;
  prev = 0;
  tmp = NULL;

  while (*p)
    {
      if ((nllCompare (*p, comma) && prev))
	{
	  prev = 0;
	}
      else
	{
	  tmp = nllAddDup (tmp, *p);
	}
      if (nllCompare (*p, rparen))
	{
	  prev = 1;
	  rc->strings = nllAdd (rc->strings, nllGlue (tmp, ""));
	  nllClear (tmp);
	  tmp = NULL;
	}

      p++;
    }
  if (tmp)
    rc->strings = nllAdd (rc->strings, nllGlue (tmp, ""));
  nllClear (tmp);
  nllClear (list);

  p = rc->strings;
  tmp = NULL;
  while (*p)
    {
      if (!nllContains (*p, lparen))
	{
	  printf ("Error:%s\n", *p);
	  rcDelete (rc);
	  return NULL;
	}
      cp = index (*p, lparen[0][0]);
      rc->names = nllAdd (rc->names, strndup (*p, cp - *p));

      //format not used here
      rc->format = nllAdd (rc->format, strdup (""));

      tmp = nllAddDup (tmp, cp);
      if (index (cp, comma[0][0]) && index (cp, dot[0][0])
	  && !index (cp, scolon[0][0]))
	rc->types = nllAddDup (rc->types, TYPE_DOUBLE);
      else if (index (cp, comma[0][0]) && !index (cp, dot[0][0])
	       && !index (cp, scolon[0][0]) && !index (cp, slash[0][0]))
	rc->types = nllAddDup (rc->types, TYPE_INTEGER);
      else if (!index (cp, comma[0][0]) && index (cp, scolon[0][0])
	       && !index (cp, slash[0][0]))
	rc->types = nllAddDup (rc->types, TYPE_CHOICE);
      else
	{
	  rcDelete (rc);
	  return NULL;
	}
      cc = index (*p, rparen[0][0]);
      rc->limits = nllAdd (rc->limits, strndup (cp + 1, cc - cp - 1));
      p++;
    }

  rc->ncoefs = nllCount (rc->names);

  p = rc->limits;
  t = rc->types;

  nllClear (tmp);
  tmp = NULL;
  while (*p)
    {
      cp = index (*p, comma[0][0]);
      if (cp != NULL)
	{			//decimals & integers
	  tmp = nllListParser (*p, comma);
	  tmp = nllDelete (tmp, comma);
	  rc->llimit = nllAddDup (rc->llimit, nllString (tmp, 0));
	  if (index (nllString (tmp, 1), slash[0][0]))
	    {			//there is one or more slash
	      char **tmp2 = NULL;
	      tmp2 = nllListParser (nllString (tmp, 1), slash);
	      tmp2 = nllDelete (tmp2, slash);
	      if (nllCount (tmp2) > 2 || strcmp (*t, "i") == 0)
		{		//more than one slash or integer
		  printf ("error in %s\n", *p);
		  nllClear (tmp);
		  nllClear (tmp2);
		  rcDelete (rc);
		  return NULL;
		}
	      rc->ulimit = nllAddDup (rc->ulimit, nllString (tmp2, 0));
	      rc->decimals = nllAddDup (rc->decimals, nllString (tmp2, 1));
	      nllClear (tmp2);
	    }
	  else
	    {			//no slash
	      rc->ulimit = nllAddDup (rc->ulimit, nllString (tmp, 1));
	      if (strcmp (*t, "i") == 0)
		{
		  rc->decimals =
		    nllAddDup (rc->decimals, DECSTRING (DECIMAL_INTEGER));
		}
	      else
		{
		  rc->decimals =
		    nllAddDup (rc->decimals, DECSTRING (DECIMAL_DOUBLE));
		}
	    }
	  rc->choice = nllAddDup (rc->choice, "");
	  nllClear (tmp);
	  tmp = NULL;
	}
      else
	{			//choice
	  tmp = nllListParser (*p, scolon);
	  tmp = nllDelete (tmp, scolon);
	  rc->llimit = nllAdd (rc->llimit, sInt2Ascii (1));
	  rc->ulimit = nllAdd (rc->ulimit, sInt2Ascii (nllCount (tmp)));
	  rc->choice = nllAdd (rc->choice, nllGlue (tmp, " "));
	  rc->decimals = nllAddDup (rc->decimals, DECSTRING (DECIMAL_CHOICE));
	  nllClear (tmp);
	  tmp = NULL;
	}
      p++;
      t++;
    }
  //nllPrint (rc->llimit, "\nllimit\n");
  //nllPrint (rc->ulimit, "\nulimit\n");
  //nllPrint (rc->decimals, "\ndecimals\n");

  rcFillOrder (rc);

  nllClear (AllStringTokens);
  return rc;
}

/**
 * analyze coefficients into a file template (inTemplate) 
 *  with @expressions and create a rangeCoefs def. 
 *  Also writes a template-analysis file (templateAnalisys).
 **/
RANGECOEFS *
rcAnalyzeTemplate (char *inTemplate, char *templateAnalisys)
{
  FILE *f;
  char *s;
  char *p, *q;
  char **refined = NULL;
  char **tmp = NULL;
  char **pp = NULL;
  char **tt = NULL;
  char **uu = NULL;

  RANGECOEFS *rc;

  int n;

  rc = rcFactory ();

  f = fopen (inTemplate, "r");
  if (f)
    {
      while ((s = InputFileLine (f)) != NULL)
	{
	  // cut at bang!
	  char *a = index (s, '!');
	  if (a != NULL)
	    *a = '\0';
	  //cut line from first '@' to last ')'
	  if ((p = index (s, '@')) != NULL)
	    {
	      q = rindex (s, ')');
	      if (q != NULL)
		{
		  rc->names = nllAdd (rc->names, strndup (p, q - p + 1));
		}
	    }
	  free (s);
	}
      fclose (f);

      if (!rc->names)
	return NULL;


//****************************************
//checks for well formed @expressions, 
// @exp without parentheses are avoided.
      pp = rc->names;
//      nllPrint(pp,"pp1=\n");
      while (*pp)
	{
	  rc->limits = nllParser (*pp, WhiteSpaceTokens);
	  tt = rc->limits;
	  while (*tt)
	    {
	      if (index (*tt, '(') != NULL || index (*tt, ')') != NULL)
		refined = nllAddDup (refined, *tt);
	      tt++;
	    }
	  nllClear (rc->limits);
	  pp++;
	}
      nllClear (rc->names);
      rc->names = refined;

//*****************************************
//split all tokens and cut at bangs (comments)
      refined = NULL;
      pp = rc->names;
//      nllPrint(pp,"pp2=\n");
      while (*pp)
	{
	  rc->limits = nllParser (*pp, TemplateTokens);
	  tt = rc->limits;
	  while (*tt)
	    {
	      refined = nllAddDup (refined, *tt);
	      tt++;
	    }
	  nllClear (rc->limits);
	  pp++;
	}
      nllClear (rc->names);


//****************************************/
      //strip out strings
      pp = refined;
//      nllPrint(pp,"pp3=\n");
      while (*pp)
	{
	  rc->names = NULL;
	  if (nllCompare (*pp, atsign))
	    {
	      while (*pp)
		{
		  rc->names = nllAddDup (rc->names, *pp);
		  if (nllCompare (*pp, rparen))
		    {
		      break;
		    }
		  pp++;
		}
	      s = nllGlue (rc->names, "");
	      rc->strings = nllAdd (rc->strings, s);
	    }
	  nllClear (rc->names);
	  pp++;
	}

      refined = nllDelete (refined, white);

//****************************************
      //strip out formats
      pp = refined;
      tmp = NULL;
      n = 0;
      while (*pp)
	{
	  rc->names = NULL;
	  //counts the number of at-expresions
	  if (nllCompare (*pp, rparen))
	    {
	      n++;
	    }
	  //checks if it has "#" sign
	  if (nllCompare (*pp, nsign))
	    {
	      while (*pp)
		{
		  //checks for a "," to finish format
		  if (nllCompare (*pp, comma))
		    {
		      break;
		    }
		  //add the token
		  rc->names = nllAddDup (rc->names, *pp);
		  pp++;
		}
	      //join the tokens into a format string
	      s = nllGlue (rc->names, "");
	      //fill up formats till now
	      for (int i = nllCount (rc->format); i < n; i++)
		rc->format = nllAddDup (rc->format, "");
	      rc->format = nllAdd (rc->format, s);
	    }
	  else
	    {
	      tmp = nllAddDup (tmp, *pp);
	    }
	  nllClear (rc->names);
	  pp++;
	}

      //must be n formats, complete it
      for (int i = nllCount (rc->format); i < n; i++)
	rc->format = nllAddDup (rc->format, "");

      nllClear (refined);
      refined = tmp;

//****************************************
      //strip out names            
      rc->names = NULL;
      rc->limits = NULL;
      pp = refined;

      while (*pp)
	{
	  if (nllCompare (*pp, atsign))
	    {
	      pp++;
	      rc->names = nllAddDup (rc->names, *pp);
	      pp++;
	      if (nllCompare (*pp, lparen))
		{
		  tmp = NULL;
		  pp++;
		  while (!nllCompare (*pp, rparen))
		    {
		      tmp = nllAddDup (tmp, *pp);
		      pp++;
		    }
		  s = nllGlue (tmp, "");
		  rc->limits = nllAddDup (rc->limits, s);
		  free (s);
		  nllClear (tmp);
		}
	      else
		{
		  printf ("Lparenthesis error:%s\n", *pp);
		  return NULL;
		}
	    }
	  else
	    {
	      if (!nllCompare (*pp, white))
		{
		  printf ("@ error: %s\n", *pp);
		  return NULL;
		}
	    }
	  pp++;
	}
      nllClear (refined);
      rc->ncoefs = nllCount (rc->names);

//****************************************
      //strip out limits
      pp = rc->limits;
      while (*pp)
	{
	  if (nllContains (*pp, scolon))
	    {
	      if (nllContains (*pp, slash))
		{
		  printf ("choice spec error: %s\n", *pp);
		  return NULL;
		}
	      rc->types = nllAddDup (rc->types, TYPE_CHOICE);
	      tmp = NULL;
	      tmp = nllListParser (*pp, scolon);
	      tmp = nllDelete (tmp, scolon);
	      n = nllCount (tmp);
	      s = nllGlue (tmp, " ");	//not | in choice
	      rc->choice = nllAddDup (rc->choice, s);
	      rc->llimit = nllAddDup (rc->llimit, "1");
	      free (s);
	      nllClear (tmp);
	      s = malloc (10);
	      sprintf (s, "%d", n);
	      rc->ulimit = nllAddDup (rc->ulimit, s);
	      free (s);
	    }
	  else
	    {
	      rc->choice = nllAddDup (rc->choice, "");
	      tmp = NULL;
	      tmp = nllListParser (*pp, comma);
	      tmp = nllDelete (tmp, comma);
	      tt = tmp;
	      rc->llimit = nllAddDup (rc->llimit, *tt);
	      tt++;
	      rc->ulimit = nllAddDup (rc->ulimit, *tt);
	      nllClear (tmp);

	      if (nllContains (*pp, dot))
		{
		  rc->types = nllAddDup (rc->types, TYPE_DOUBLE);
		}
	      else
		{
		  if (nllContains (*pp, slash))
		    {
		      printf ("integer spec error:%s\n", *pp);
		      return NULL;
		    }
		  rc->types = nllAddDup (rc->types, TYPE_INTEGER);
		}
	    }
	  pp++;
	}

//****************************************
      tt = NULL;
      pp = rc->ulimit;
      rc->decimals = NULL;
      uu = rc->types;
      while (*pp)
	{
	  /// it is a double?
	  if (strcmp (*uu, "d") == 0)
	    {			//yes
	      if (nllContains (*pp, slash))
		{
		  tmp = NULL;
		  tmp = nllListParser (*pp, slash);
		  //check for one slash in decimal part
		  n = nllCount (tmp);
		  s = nllString (tmp, 2);
		  if (n != 3 || strtol (s, NULL, 10) < 1)
		    {		//more than one slash
		      printf ("decimals spec error:%s\n", *pp);
		      return NULL;
		    }
		  tt = nllAddDup (tt, nllString (tmp, 0));
		  rc->decimals = nllAddDup (rc->decimals, s);
		}
	      else
		{
		  //it has not a slash
		  tt = nllAddDup (tt, *pp);
		  rc->decimals = nllAddDup (rc->decimals, "9");
		}
	    }
	  else
	    {
	      //integer type (integer plus choice)
	      tt = nllAddDup (tt, *pp);
	      rc->decimals = nllAddDup (rc->decimals, "0");
	    }
	  pp++;
	  uu++;
	}
      nllClear (rc->ulimit);
      rc->ulimit = tt;
      //rcPrint (rc);

      rcFillOrder (rc);

      if (rcWriteAnalysisResults (rc, templateAnalisys) == 0)
	return NULL;
      return rc;
    }

  else
    {
      printf ("No file: %s\n", inTemplate);
    }

  return NULL;
}

int
rcWriteResponse (RANGECOEFS * rc, char *etype,
		 char *einput, char *efit, char *bounds)
{
  FILE *w;
  int i;
  w = fopen ("response", "w");
  if (!w)
    {
      printf ("Cannot open file 'response'\n");
      return 0;
    }
  fprintf (w, "\n");
  fprintf (w, "[job]\n");
  fprintf (w, "type: %s\n", etype);
  fprintf (w, "coefficients: %d\n", rc->ncoefs);
  fprintf (w, "external input: %s\n", einput);
  fprintf (w, "external fit: %s\n", efit);
  fprintf (w, "bounds: %s\n", bounds);
  fprintf (w, "\n");
  if (rc->names)
    {
      fprintf (w, "[coefficient names]\n");
      for (i = 0; i < rc->ncoefs; i++)
	fprintf (w, "%s\n", rc->names[i]);
    }
  fclose (w);
  return 1;
}


int
rcWriteBounds (RANGECOEFS * rc, char *bounds, char *origintemplate)
{
  FILE *w;
  char **l, **u, **d;
  w = fopen (bounds, "w");
  fprintf (w, "Automatic build from %s\n", origintemplate);
  l = rc->llimit;
  u = rc->ulimit;
  d = rc->decimals;
  while (*l)
    {
      fprintf (w, "\t%s\t%s\t%s\n", *l, *u, *d);
      l++;
      u++;
      d++;
    }
  fclose (w);
  return 1;
}

void
rcFormat (char *format, char t, char *s)
{
  char *p, *pp;
  int width, prec;
  if (!format || strcmp (format, " ") == 0)
    {
      switch (t)
	{
	case 'i':
	  strcpy (s, "%d");
	  break;
	case 'd':
	case 'c':
	  strcpy (s, "%f");
	  break;
	default:
	  printf ("error rcformat\n");
	}
      return;
    }
  p = strdup (format);
  p = sRtrim (sLtrim (p));
  width = strlen (p);
  if ((pp = index (p, '.')) != NULL)
    {
      prec = width - (pp - p) - 1;
      snprintf (s, AT_LINE_MAX - 1, "%%%d.%d", width, prec);
    }
  else
    {
      snprintf (s, AT_LINE_MAX - 1, "%%%d", width);
    }
  switch (t)
    {
    case 'i':
      strcat (s, "d");
      break;
    case 'd':
    case 'c':
      strcat (s, "f");
      break;
    default:
      printf ("error rcformat '%c' %s\n", t, format);
    }

  free (p);
}

/**
 * fill in the parameters into template text
 * @parameter text, the template text
 * @parameter coefs, the coefficients
 * @parameter rc, a RANGECOEFS struct with specifications
 * */
//*********************************************************
#define rco(x) atoi(rc->coefs[x])-1
//*********************************************************

char *
rcCreateParameters (char *text, double *coefs, RANGECOEFS * rc)
{
  char tmp[AT_LINE_MAX];
  char rtmp[AT_LINE_MAX];	//negative value

  char tmp2[AT_LINE_MAX];

  char negative[AT_LINE_MAX];	//expression to replace

  char xformat[AT_LINE_MAX];
  int i, k;
  char *s;
  char **cc;
  double dtmp;

  for (i = 0; i < rc->ncoefs; i++)
    {
      s = nllString (rc->types, i);
      rcFormat (nllString (rc->format, i), s[0], xformat);
      switch (s[0])
	{
	case 'i':
	  k = coefs[rco (i)];
	  snprintf (tmp, AT_LINE_MAX, xformat, k);
	  snprintf (rtmp, AT_LINE_MAX, xformat, -k);
	  break;
	case 'd':
	  snprintf (tmp, AT_LINE_MAX, xformat, coefs[rco (i)]);
	  snprintf (rtmp, AT_LINE_MAX, xformat, -coefs[rco (i)]);
	  break;
	case 'c':
	  cc = NULL;
	  strncpy (tmp2, nllString (rc->choice, i), AT_LINE_MAX);
	  cc = nllListParser (tmp2, white);
	  cc = nllDelete (cc, white);
	  k = coefs[rco (i)] - 1;
	  if (sStr2Double (nllString (cc, k), &dtmp))
	    {
	      snprintf (tmp, AT_LINE_MAX, xformat, dtmp);
	      snprintf (rtmp, AT_LINE_MAX, xformat, -dtmp);
	      nllClear (cc);
	    }
	  else
	    {
	      printf ("error choice format\n");
	      return NULL;
	    }

	  break;
	default:
	  printf ("CreateParameters error\n");
	  return NULL;
	}
      //replace '-@name(...)'
      snprintf (negative, AT_LINE_MAX, "-%s", nllString (rc->strings, i));
      text = sReplaceString (text, negative, rtmp);
      //replace '@name(...)'
      text = sReplaceString (text, nllString (rc->strings, i), tmp);
      //replace '-@name'
      strcpy (tmp2, "-@");
      strcat (tmp2, nllString (rc->names, i));
      text = sReplaceString (text, tmp2, rtmp);
      //replace '@name'
      strcpy (tmp2, "@");
      strcat (tmp2, nllString (rc->names, i));
      text = sReplaceString (text, tmp2, tmp);
    }
  return text;
}

int
varlen (char *s)
{
  return index (s, '|') - s;
}

/**
 * read analysis results into memory
 * @parameter arfile the analysis results file
 * @return a RANGECOEFS structure
 **/
RANGECOEFS *
rcReadAnalysisResults (char *arfile)
{
  FILE *r;
  char *s, **p;
  int n;
  char **list = NULL;
  RANGECOEFS *rc;
  rc = rcFactory ();
  p = NULL;
  n = 0;
  r = fopen (arfile, "r");
  if (r)
    {
      while ((s = InputFileLine (r)) != NULL)
	{
	  if (!isBlankLine (s))
	    list = nllAdd (list, s);
	}

      n = nllCount (list);
      list = nllAnyOrder (list, varlen);

      for (int i = 0; i < n; i++)
	{
	  s = nllStringDup (list, i);
	  //check two vbar->empty!
	  s = sReplaceString (s, "||", "| |");
	  p = nllListParser (s, vbar);
	  p = nllDelete (p, vbar);
	  rcAddList (rc, p);
	  nllClear (p);
	  p = NULL;
	  free (s);
	}
      fclose (r);
      rc->ncoefs = n;
      nllClear (list);
      return rc;
    }
  printf ("Error reading file: %s\n", arfile);
  return NULL;
}


char *
rcGetChoice (int i, int value, RANGECOEFS * rc)
{
  char **list = NULL;
  char *ret;
  list = nllListParser (nllString (rc->choice, i), white);
  list = nllDelete (list, white);
  ret = strdup (nllString (list, value));
  nllClear (list);
  return ret;
}

const char *
rcGetType (int i, RANGECOEFS * rc)
{
  return nllString (rc->types, i);
}
