/*
(c) GAFit toolkit $Id: nllist.c 500 2024-12-19 21:36:14Z ro $
*/
#if HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <stdarg.h>
#include "nllist.h"

//print on standard output a list.
void
nllPrint (char **list, char *id)
{
  char **p = list;
  printf ("%s", id);
  if (!p)
    {
      printf ("NULL\n");
      return;
    }
  while (*p)
    {
      printf ("[%s]", *p);
      p++;
    }
}

//pops from list to another list, working both as stacks
char **
nllPopPush (char ***listFrom, char **listTo)
{
  char *u;
  u = nllPop (listFrom);
  listTo = nllPush (listTo, u);
  free (u);			//special case (char *u must be freed)
  return listTo;
}

//reverse stack
char **
nllReverse (char **list)
{
  char **rlist = NULL;
  char *v;
  if (list)
    {
      v = nllLook (list);
      while (v)
	{
	  rlist = nllPopPush (&list, rlist);
	  v = nllLook (list);
	}
      nllClear (list);
    }
  return rlist;
}

//special case: delete top of list working as stack 
void
nllPopLost (char ***list)
{
  char *u;
  u = nllPop (list);
  free (u);
}

//delete duplicates in list
char **
nllUnic (char **list)
{
  char **i = list;
  char **newlist = NULL;
  if (list)
    {
      while (*i)
	{
	  if (!nllCompare (*i, newlist))
	    {
	      newlist = nllAdd (newlist, strdup (*i));
	    }
	  i++;
	}
      nllClear (list);
      return newlist;
    }
  else
    return list;
}

//order by length
char **
nllLenOrder (char **list)
{
  char *tmp;
  int i, j;
  int tlen = nllCount (list);

  for (i = 0; i < tlen; i++)
    for (j = i + 1; j < tlen; j++)
      if (strlen (list[i]) < strlen (list[j]))
	{
	  tmp = list[j];
	  list[j] = list[i];
	  list[i] = tmp;
	}
  return list;
}

//order by any 
char **
nllAnyOrder (char **list, int func (char *))
{
  char *tmp;
  int i, j;
  int tlen = nllCount (list);

  for (i = 0; i < tlen; i++)
    for (j = i + 1; j < tlen; j++)
      if (func (list[i]) < func (list[j]))
	{
	  tmp = list[j];
	  list[j] = list[i];
	  list[i] = tmp;
	}
  return list;
}


//delete selected items (list deleteSet)  from list
char **
nllDelete (char **list, char **deleteSet)
{
  char **p, **result = NULL;
  p = list;
  while (*p)
    {
      if (!nllCompare (*p, deleteSet))
	result = nllAdd (result, strdup (*p));
      p++;
    }
  nllClear (list);
  return result;
}

//delete from markset to markset
char **
nllDelMark2Mark (char **list, char **from, char **to)
{
  char **a;
  char **nwlist = NULL;
  int mark = 0;
  a = list;
  while (*a)
    {
      if (nllCompare (*a, from))
	mark = 1;
      if (!mark)
	nwlist = nllAdd (nwlist, strdup (*a));
      if (nllCompare (*a, to))
	mark = 0;
      a++;
    }
  nllClear (list);
  return nwlist;
}

//checks if a string is in a list
int
nllCompare (char *p, char **list)
{
  char **t = list;
  if (t)
    while (*t)
      {
	if (!nllStrCmp (p, *t))
	  return 1;
	t++;
      }
  return 0;
}

//checks if any of the tokens in the list is in a string
int
nllContains (char *p, char **list)
{
  char **t = list;
  if (t)
    while (*t)
      {
	if (strstr (p, *t) != NULL)
	  return 1;
	t++;
      }
  return 0;
}

//push onto list working as stack
char **
nllPush (char **where, char *what)
{
  return nllAdd (where, strdup (what));
}

//pops last from where list working as stack
char *
nllPop (char ***where)
{
  char **p = *where;
  char *ret;
  int n = 0;
// novo
  if (!p)
    return NULL;
  if (!(*p))
    return NULL;
  while (*p++)
    n++;
  p = *where;
  ret = p[n - 1];
  p[n - 1] = NULL;
  p = realloc (p, sizeof (char *) * (n));
  *where = p;
  return ret;
}

//check last in "where" list working as stack
char *
nllLook (char **where)
{
  if (!where)
    return NULL;
  if (!(*where))
    return NULL;
  while (*where)
    where++;
  where--;
  return *where;
}

// replace "*where" with "with".
// 
void
nllItemReplace (char **where, char *with)
{
  int n = strlen (with);
  char *w = malloc (n + 1);
  free (*where);
  strcpy (w, with);
  *where = w;
}

/*
 *  char replace in "where" all "what" with "with"
 * @param where the null terminated list to process
 * @param what the string to replace
 * @param with the string to replace with
 *
 * @returns the null terminated list replaced
 */

char **
nllReplace (char **where, char *what, char *with)
{
  char **s = where;
  //printf ("reemplazar [%s] con [%s] \n", what, with);
  where++;
  while (*where)
    {
      if (!strcmp (*where, what))
	nllItemReplace (where, with);
      where++;
    }
  return s;
}

//returns index of v in list.
int
nllLocate (char *v, char **list)
{
  int i = 0;
  while (*list)
    {
      if (!nllStrCmp (v, *list))
	return i;
      list++;
      i++;
    }
  return -1;
}

//returns item nth from list
char *const
nllString (char **list, int n)
{
  int count = 0;
  while (*list)
    {
      if (count == n)
	return *list;
      count++;
      list++;
    }
  return NULLIST__EMPTY_STRING__;
}

//returns a COPY of item nth from list
char *
nllStringDup (char **list, int n)
{
  int count = 0;
  while (*list)
    {
      if (count == n)
	return strdup (*list);
      count++;
      list++;
    }
  return strdup (NULLIST__EMPTY_STRING__);
}

/* slow
char *const
nllString (char **list, int n)
{
  int count = nllCount(list);
  if(n<0 || count==0 || n>=count)
  return NULLIST__EMPTY_STRING__;
  else
  return list[n];
}
*/


//returns count of items in list
int
nllCount (char **list)
{
  int i = 0;
  if (!list)
    return 0;
  while (*list++)
    i++;
  return i;
}

//erase a list
void
nllClear (char **tk)
{
  char **pk = tk;
  // novo
  if (!pk)
    return;
  while (*pk)
    free (*pk++);
  //last must be NULL.
  assert (*pk == NULL);
  free (tk);
}

//string compare
int
nllStrCmp (char *s1, char *s2)
{
  if (s1 == s2)
    return 0;
  if (!s1 || !s2)
    return 1;
  return strcmp (s1, s2);
}

//add must be a copy of original string!
char **
nllAdd (char **list, char *add)
{
  int n;
  char **p;
  if (!list)
    {
      list = malloc (sizeof (char *));
      list[0] = NULL;
    }
  if (!add)
    return list;
  p = list;
  n = 1;
  while (*p++)			//counting elements
    n++;
  list = realloc (list, sizeof (char *) * (n + 1));
  list[n] = NULL;
  list[n - 1] = add;
  return list;
}

//add a copy of original string!
char **
nllAddDup (char **list, char *add)
{
  return nllAdd (list, strdup (add));
}

//parse a string getting tokens created from external utset function
char **
nllParser (char *l, char **(*utset) (void))
{
  char *p, *ap;
  char **t = NULL;
  char **to;
  int inc;
  ap = p = l;
  while (*p)
    {
      to = utset ();
      inc = 1;
      while (*to)
	{
	  if (!strncmp (p, *to, strlen (*to)))
	    {
	      inc = strlen (*to);
	      if (p != ap)
		t = nllAdd (t, strndup (ap, p - ap));
	      ap = p + inc;
	      t = nllAdd (t, strndup (*to, strlen (*to)));
	      break;;
	    }
	  to++;
	}
      p = p + inc;
    }
  if (p != ap)
    t = nllAdd (t, strndup (ap, p - ap));
  return t;
}

/*
 * /parse a string getting tokens from a null list
 * @parameter l the string to parse
 * @parameter utset the null list tokens
 */

char **
nllListParser (char *l, char **utset)
{
  char *p, *ap;
  char **t = NULL;
  char **to;
  int inc;
  ap = p = l;
  while (*p)
    {
      to = utset;
      inc = 1;
      while (*to)
	{
	  if (!strncmp (p, *to, strlen (*to)))
	    {
	      inc = strlen (*to);
	      if (p != ap)
		t = nllAdd (t, strndup (ap, p - ap));
	      ap = p + inc;
	      t = nllAdd (t, strndup (*to, strlen (*to)));
	      break;
	    }
	  to++;
	}
      p = p + inc;
    }
  if (p != ap)
    t = nllAdd (t, strndup (ap, p - ap));
  return t;
}


/* build a null terminated list from a list of strings:
 * nllBuild("hello","hi",...,NULL) <-last one must be NULL.
 * @return returns the null terminated list.
 */

char **
nllBuild (const char *first, ...)
{
  char *p;
  char **l = NULL;
  va_list args;

  if (first == NULL)
    return NULL;
  va_start (args, first);
  l = nllAddDup (l, (char *) first);
  while ((p = va_arg (args, char *)) != NULL)
      l = nllAddDup (l, p);
  va_end (args);
  return l;
}


//concatenate a list with glue
char *
nllGlue (char **list, char *glue)
{
  char *s = NULL;
  int len = 0;
  int glen = strlen (glue);
  char **p = list;
  while (*p)
    {
      if (*p)
	{
	  len += strlen (*p) + glen;
	}
      p++;
    }
  s = realloc (s, len + 1);

  s[0] = 0;
  p = list;

  if (*p)
    strcat (s, *p);
  p++;
  while (*p)
    {
      strcat (s, glue);
      strcat (s, *p);
      p++;
    }
  return s;
}

int
nllCharEmpty (char *c)
{
  return !strcmp (NULLIST__EMPTY_STRING__, c);
}

// new list from list + list
char **
nllListUnion (char **l1, char **l2)
{
  char **l3 = NULL;
  char **p = l1;
  while (*p)
    {
      l3 = nllAdd (l3, strdup (*p));
      p++;
    }
  p = l2;
  while (*p)
    {
      if (!nllCompare (*p, l3))
	l3 = nllAdd (l3, strdup (*p));
      p++;
    }
  return nllUnic (l3);
}

// new list from intersection of l1 and l2
char **
nllListInter (char **l1, char **l2)
{
  char **l3 = NULL;
  char **p = l1;
  while (*p)
    {
      if (nllCompare (*p, l2))
	l3 = nllAdd (l3, strdup (*p));
      p++;
    }
  return l3;
}

// new list from l1 minus l2
char **
nllListSubs (char **l1, char **l2)
{
  char **l3 = NULL;
  char **p = l1;
  while (*p)
    {
      if (!nllCompare (*p, l2))
	l3 = nllAdd (l3, strdup (*p));
      p++;
    }
  return l3;

}

// duplicate list
char **
nllListDup (char **l2)
{
  char **l1 = NULL;
  char **p = l2;
  while (*p)
    {
      l1 = nllAdd (l1, strdup (*p));
      p++;
    }
  return l1;
}

// add to l1 items from l2
char **
nllListAdd (char **l1, char **l2)
{
  char **p = l2;
  while (*p)
    l1 = nllAddDup (l1, *p++);
  return l1;
}

// sub list from to. From begins in "1"
char **
nllSubList (char **l, int from, int to)
{
  int len = nllCount (l);
  char **l1 = NULL;
  if (from > to || from > len)
    return NULL;
  if (to > len)
    to = len;
  if (from < 1)
    from = 1;
  for (int i = from - 1; i < to; i++)
    {
      l1 = nllAddDup (l1, l[i]);
    }
  return l1;
}
