/*
 * (c) GAFit toolkit $Id: mvcommon.c 510 2025-04-22 14:23:13Z ro $
 */
#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <math.h>


#include "mvcommon.h"
#include "../rangecf/rangecf.h"
#include "../inputline/line.h"
#include "../flyctl/flyctl.h"
#include "../nullist/nllist.h"
#include "../parameters/parameters.h"
#include "../rstrings/rstrings.h"
#include "../compiler/ucompiler.h"
#include "../fpu/fpu.h"
#include "../core.h"

static FPU2011 *fpu;

extern char *comma[];
extern char *white[];

//cached results
char **AllColumnsTokens = NULL;

char **
ColumnsTokens (void)
{
  char **s = NULL;

  if (AllColumnsTokens)
    return AllColumnsTokens;

  s = nllBuild (COMMA, SPACE, TAB, NULL);

  AllColumnsTokens = nllLenOrder (s);
  return AllColumnsTokens;
}


char *
mvGetParameters (char *job,
		 char *coefficients, char *fitvar, char *datafile,
		 char *columns, char *expression, int *headers, int *pasm,
		 int maxstr)
{
  char *l = NULL;
  *headers = GetIntParameter (job, "multi variable", "data headers", 0);
  *pasm = GetBoolParameter (job, "multi variable", "print asm code", 0);
  GetStringParameter (job, "multi variable", "data file", "none",
		      datafile, maxstr);
///> infact "parameters" are merit function coefficients.
  if (!GetStringParameter (job, "multi variable", "coefficients", "STOP",
			   coefficients, maxstr) &&
      !GetStringParameter (job, "multi variable", "parameters", "STOP",
			   coefficients, maxstr))
    {
      finito (" coefficients");
    }
  else
    {
      GetStringParameter
	(job, "multi variable", "fit variable", FIT_VARIABLE, fitvar, maxstr);
      GetStringParameter (job, "multi variable", "expression",
			  EXPRESSION_SECTION, expression, maxstr);
      if (strcmp (datafile, "none") != 0)
	{
	  if (!GetStringParameter (job, "multi variable", "data columns",
				   "STOP", columns, maxstr))
	    {
	      finito ("Error data columns\n");
	    }
	}
    }

  l = SectionText (job, expression);

  return l;
}

int
mvGetJobParameters (char *file, char *einput, char *efit, char *ebounds)
{
  int jobtype;
  char *jtype[] = LIST_JOB_TYPES;
  jobtype =
    GetFromSetParameter (file, "job", "type", jtype, JOB_TYPE_AUTO, 0);

  GetStringParameter (file, "job", "external input", "external.input",
		      einput, MAXSTR);
  GetStringParameter (file, "job", "external fit", "external.fit",
		      efit, MAXSTR);
  GetStringParameter (file, "job", "bounds", "bounds.txt", ebounds, MAXSTR);
  if (jobtype == JOB_TYPE_AUTO)
    return 1;
  else
    return 0;
}


void
mvPrintPrettyExpr (char *l)
{
  char *p = l;
  int n;
  printf ("\n\t");
  n = 0;
  do
    {
      switch (*p)
	{
	case ';':
	  printf (";\n\n\t");
	  n = 0;
	  break;
	case ' ':
	case '\n':
	  break;
	case '+':
	case '-':
	  printf (" %c ", *p);
	  if (n > 60)
	    {
	      printf ("\n\t");
	      n = 0;
	    }
	  break;
	case '*':
	case '=':
	case '/':
	  printf (" %c ", *p);
	  break;
	default:
	  if (*p)
	    printf ("%c", *p);
	}
      n++;
    }
  while (*p++);
  printf ("\n\n");
}

void
mvPrintPrettyCoeffs (char *l)
{
  char *p = l;
  int n, isclose;
  printf ("\n\t");
  n = 0;
  isclose = 1;
  do
    {
      switch (*p)
	{
	case ' ':
	case '\n':
	  break;
	case '(':
	  printf ("( ");
	  isclose = 0;
	  break;
	case ')':
	  printf (" )");
	  isclose = 1;
	  break;
	case ';':
	  printf ("; ");
	  break;
	case ',':
	  printf (", ");
	  if (n > 60 && isclose)
	    {
	      printf ("\n\t");
	      n = 0;
	    }
	  break;
	default:
	  if (*p)
	    printf ("%c", *p);
	}
      n++;
    }
  while (*p++);
}

void
finito (char *txt)
{
  printf ("Error: %s\n", txt);
  fcStopIt (txt);
}

void
mvSaveConfiguration (char *einput, char *efit, char *fitvar, char **prgmvars,
		     char **varstype, char **datavars)
{
  FILE *w;
//  char **p;
  char *s;

  w = fopen (CONFIGURATION_FILE, "w");
  if (w != NULL)
    {
      fprintf (w, "%s\n", einput);
      fprintf (w, "%s\n", efit);
      fprintf (w, "%s\n", fitvar);
      s = nllGlue (prgmvars, " ");
      fprintf (w, "%s\n", s);
      free (s);

      s = nllGlue (varstype, " ");
      fprintf (w, "%s\n", s);
      free (s);

      s = nllGlue (datavars, " ");
      fprintf (w, "%s\n", s);
      free (s);

      fclose (w);
    }
  else
    {
      finito ("Error: cannot open configuration file");
    }
}

int
loadOne (FILE * f, char *where)
{
  char *s;
  if ((s = InputFileLine (f)) != NULL)
    {
      strncpy (where, s, MAXSTR - 1);
      free (s);
      return 1;
    }
  return 0;
}

void
mvReadConfiguration (char *einput, char *efit, char *fitvar, char ***p,
		     char ***v, char ***d)
{
  FILE *r;
  char s[MAXSTR];

  r = fopen (CONFIGURATION_FILE, "r");
  if (r != NULL)
    {
      if (loadOne (r, einput))
	{
	  if (loadOne (r, efit))
	    {
	      if (loadOne (r, fitvar))
		{
		  if (loadOne (r, s))
		    {
		      *p = nllListParser (s, white);
		      *p = nllDelete (*p, white);
		      if (loadOne (r, s))
			{
			  *v = nllListParser (s, white);
			  *v = nllDelete (*v, white);
			  if (loadOne (r, s))
			    {
			      *d = nllListParser (s, white);
			      *d = nllDelete (*d, white);
			      fclose (r);
			      return;
			    }
			}
		    }
		}
	    }
	}
    }
  finito ("Error reading configuration file");
}

int
isTypeNumber (char *s)
{
  return (strcmp (s, "i") != 0 && strcmp (s, "d") != 0 && strcmp (s, "c") != 0
	  && strcmp (s, "x") != 0);
}

int
isTypeCoefficient (char *s)
{
  return (strcmp (s, "i") == 0 || strcmp (s, "d") == 0
	  || strcmp (s, "c") == 0);
}

int
isTypeChoice (char *s)
{
  return (strcmp (s, "c") == 0);
}

double
mvEvaluateOne (double *variables, int nvars, int nfit)
{
  int flags;
  double fit;
  fpuMemoryBlockAdd (fpu, variables, nvars);
  //run calculation  

  flags = fpu2011 (fpu);
  if (flags)
    {
      printf ("\nERROR:\t");
      fpuExplainStatus (flags);
      printf ("\n");
      finito ("Calculation error");
    }

  //get fit value after calculation
  fit = fpuGetVariable (fpu, nfit);
//  free (variables);

  fpuReset (fpu);
  //fpu ready to run again.

  return fit;
}

void
showVars (int intvar, double *variables, char **pvar, char **types)
{
  for (int k = 0; k < intvar; k++)
    {
      printf ("\t%s \t\t= \t%f \t(%s)\n", pvar[k], variables[k], types[k]);

    }
}

void
mvMachineGun (int n, RANGECOEFS * rc, char *einput, char *efit,
	      char *fitvar, char **pvar, char **type, char **dvar, int test)
{
  FILE *e, *f;
  int i, j, k;
  char *s;
  int columns, lines;
  double coefs[rc->ncoefs];
  double fit, ofit;
  int intvar = nllCount (pvar);
  double variables[intvar];
  char *binary;
  int nc, nfit;
  char **tmp;

  binary = MakePath (BINARY_NAME, ".uxe");

  //load data
  e = fopen (PARSED_DATA_FILE, "r");
  if (e)
    {
      fscanf (e, "%d %d", &lines, &columns);
      {
	double data[lines][columns];

	fpu = fpuNew ();
	fpuLoad (fpu, binary);

	for (i = 0; i < lines; i++)
	  for (j = 0; j < columns; j++)
	    fscanf (e, "%lf", &data[i][j]);
	fclose (e);
	//lookup for fit variable

	nfit = nllLocate (fitvar, pvar);

	//load ga vectors
	f = fopen (einput, "r");

	//open external fit file
	if (!test)
	  e = fopen (efit, "w");

	if (f)
	  {
	    for (i = 0; i < n; i++)
	      {
		for (j = 0; j < rc->ncoefs; j++)
		  {
		    s = InputFileLine (f);
		    if (!s)
		      {
			finito ("Error loading ga vectors");
		      }
		    if (test)
		      {
			if (test == 1)
			  {
			    sscanf (s, "%lf", &coefs[j]);
			  }
			else
			  {
			    int ind;
			    tmp = nllListParser (s, white);
			    tmp = nllDelete (tmp, white);
			    if ((ind = nllLocate (tmp[0], rc->names)) < 0)
			      finito ("unknown coefficient");
			    sscanf (tmp[1], "%lf", &coefs[ind]);
			    nllClear (tmp);
			  }
		      }
		    else
		      {
			sscanf (s, "%lf", &coefs[j]);
		      }
		    free (s);
		  }
		//blank line
		s = InputFileLine (f);
		free (s);

		//one individual loaded     
		fit = 0.0;

		//load coefficients
		nc = 0;
		for (j = 0; j < intvar; j++)
		  {
		    if (isTypeCoefficient (type[j]))
		      {
			//coefficient load
			if (!isTypeChoice (type[j]))
			  {
			    variables[j] = coefs[nc];
			  }
			else
			  {
			    //choice coefficient replace
			    tmp = nllListParser (rc->choice[nc], white);
			    tmp = nllDelete (tmp, white);
			    if (!sStr2Double
				(nllString (tmp, coefs[nc] - 1),
				 &variables[j]))
			      {
				finito ("Error: cannot convert choice type");
			      }
			    nllClear (tmp);
			  }
			nc++;
		      }
		    else
		      {
			//set to zero the rest of variables
			variables[j] = 0;
		      }
		  }

		for (j = 0; j < lines; j++)
		  {

		    //load data
		    nc = 0;
		    for (k = 0; k < intvar; k++)
		      {
			if (isTypeNumber (type[k]))
			  {
			    variables[k] = data[j][nc];
			    nc++;
			  }
		      }
#ifdef __DEBUG__
		    showVars (intvar, variables, pvar, type);
#endif
		    ofit = mvEvaluateOne (variables, intvar, nfit);
		    fit += ofit;
		    if (test)
		      printf ("point %5d: %20.5lf\n", j, ofit);
#ifdef __DEBUG__
		    showVars (intvar, variables, pvar, type);
		    printf ("\t\tsum of fits:%lf\n", fit);
#endif
		  }
		fprintf (e, "%lf\n", fit);
	      }
	    fclose (f);

	    if (!test)
	      {
		fclose (e);
	      }
	    else
	      {
		printf ("Fitness: %20.5f\n", fit);
	      }

	    free (binary);
	  }
      }
    }
  return;
}


char **
ClassifyVars (char **prgmvars, RANGECOEFS * rc, char **datavariables)
{
  char **p;
  char **types = NULL;
  char tmp[MAXSTR];
  int in;
  p = prgmvars;
  while (*p)
    {
      if ((in = nllLocate (*p, rc->names)) >= 0)
	{
// assign types to coefficients      
	  types = nllAddDup (types, nllString (rc->types, in));
	}
      else if ((in = nllLocate (*p, datavariables)) >= 0)
	{
// assing column from data file to data variables       
	  sprintf (tmp, "%d", in + 1);
	  types = nllAddDup (types, tmp);
	}
      else
	{
// intermediate variables       
	  types = nllAddDup (types, "x");
	}
      p++;
    }
  return types;
}


void
mvParseData (char *data, int dh, char **dvar, char **vtype)
{

  FILE *e;
  int i, j;
  int columns, lines, fcolumns;
  char *s;
  char **tmp;
  char ***filedata;
  //parse data file
  columns = nllCount (dvar);
  filedata = malloc (sizeof (char **) * columns);
  for (i = 0; i < columns; i++)
    filedata[i] = NULL;
  e = fopen (data, "r");
  //skip headers
  for (i = 0; i < dh; i++)
    {
      s = InputFileLine (e);
      free (s);
    }

  lines = 0;
  while ((s = InputFileLine (e)))
    {
      if (isBlankLine (s))
	{
	  free (s);
	  continue;
	}

      lines++;
//      printf ("%d - %s\n", lines, s);

      tmp = nllListParser (s, white);
      tmp = nllDelete (tmp, white);
      for (i = 0; i < columns; i++)
	{
	  filedata[i] = nllAddDup (filedata[i], nllString (tmp, i));
	}
      nllClear (tmp);
      free (s);
    }
  fclose (e);
//count columns needed
  tmp = vtype;
  fcolumns = 0;
  while (*tmp)
    {
      while (*tmp)
	{
	  if (isTypeNumber (*tmp))
	    {
	      fcolumns++;
	    }
	  tmp++;
	}
    }

  e = fopen (PARSED_DATA_FILE, "w");
  if (e)
    {
      fprintf (e, "%d\t%d\n", lines, fcolumns);
      for (i = 0; i < lines; i++)
	{
	  tmp = vtype;
	  while (*tmp)
	    {
	      if (isTypeNumber (*tmp))
		{
		  j = atoi (*tmp) - 1;
		  fprintf (e, "\t%s", filedata[j][i]);
		}
	      tmp++;
	    }
	  fprintf (e, "\n");;
	}
      for (i = 0; i < columns; i++)
	nllClear (filedata[i]);
      free (filedata);
      fclose (e);
    }
  else
    {
      finito ("Cannot write parsed data file");
    }

}
