/*
   (c) GAFit toolkit $Id: regpots.c 551 2025-09-23 19:43:48Z ro $
*/
#if HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "../core.h"
#include "../flyctl/flyctl.h"
#include "../nullist/nllist.h"
#include "regpots.h"

/* 
 * structure to store 
 * the known potentials
 */
POTS knownPots;

/*
 * The selected potential or -1 (none)
 */
int ActualPot;
int Actual3Pot;

/*
 * initialize the potentials list
 */
void
InitPot ()
{
  ActualPot = -1;
  Actual3Pot = -1;
  knownPots.n = 0;
  knownPots.name = NULL;
  knownPots.npot = NULL;
  knownPots.ncoefs = NULL;
  knownPots.lang = NULL;
  knownPots.funcs = NULL;
  FortranSetupPots ();
  CSetupPots ();
}

/*
 * register one potential
 * @param name the name
 * @param npot potential number (fortran)
 * @param ncoefs number of coefficients
 * @param lang the language (fortran,C, etc.)
 * @param f potentials C function pointer
 * @param type '2' or '3' body pot
 */
void
regOne (char *name, int npot, int ncoefs, char lang, FPOINTER f, char type)
{
  knownPots.n++;
  knownPots.name =
    (char **) realloc (knownPots.name, sizeof (char *) * knownPots.n);
  knownPots.name[knownPots.n - 1] = name;
  knownPots.npot =
    (int *) realloc (knownPots.npot, sizeof (int) * knownPots.n);
  knownPots.npot[knownPots.n - 1] = npot;
  knownPots.ncoefs =
    (int *) realloc (knownPots.ncoefs, sizeof (int) * knownPots.n);
  knownPots.ncoefs[knownPots.n - 1] = ncoefs;
  knownPots.lang =
    (char *) realloc (knownPots.lang, sizeof (char) * knownPots.n);
  knownPots.lang[knownPots.n - 1] = lang;
  knownPots.kind =
    (char *) realloc (knownPots.kind, sizeof (char) * knownPots.n);
  knownPots.kind[knownPots.n - 1] = type;
  knownPots.funcs =
    realloc (knownPots.funcs, sizeof (FPOINTER) * knownPots.n);
  knownPots.funcs[knownPots.n - 1] = f;
}

/*
 * Register a fortran potential
 * @param name the name
 * @param npot fortran potential number
 * @param ncoefs number of coefficients
 * @param type '2' or '3'
 * @param n the length of the name's string.
 * @param m the length of the type string.
 *
 */
void
FRegisterPot (char *name, int *npot, int *ncoefs, char *type, long n, long m)
{
  char *cpy = strndup (name, n);
  char tt = type[0];
  regOne (cpy, *npot, *ncoefs, 'f', NULL, tt);
}

/*
 * Register a C potential
 * @param name the name
 * @param f C function pointer
 * @param ncoefs number of coefficientsA
 * @param type '2' or '3' body potential
 */
void
CRegisterPot (char *name, FPOINTER f, int ncoefs, char type)
{
  char *cpy = strdup (name);
  regOne (cpy, 0, ncoefs, 'c', f, type);
}

/*
 * Prints the known potentials
 */
void
printPots (void)
{
  printf ("\n");
  printf ("  KNOWN POTENTIALS\n");
  printf
    ("+--------+--------+----+---------------------+-------------+--------+\n");
  printf
    ("| number | coefs  |lang|name                 | function    |  kind  |\n");
  printf
    ("+--------+--------+----+---------------------+-------------+--------+\n");
  for (int i = 0; i < knownPots.n; i++)
    {
      printf ("|%6d  |%6d  |  %c | %-20s|", i + 1,
	      knownPots.ncoefs[i], knownPots.lang[i], knownPots.name[i]);

      switch (knownPots.lang[i])
	{
	case 'f':
	  printf (" %7d     |", knownPots.npot[i]);
	  break;
	case 'c':
	  printf (" %11p |", knownPots.funcs[i]);
	  break;
	}

      switch (knownPots.kind[i])
	{
	case '2':
	  printf (" %6s |\n", "2body");
	  break;
	case '3':
	  printf (" %6s |\n", "3body");
	  break;
	}

    }
  printf
    ("+--------+--------+----+---------------------+-------------+--------+\n");
  printf ("\n");
}

/* list of known potentials
 * @parameter kind, potential type, '2' or '3'
 * @return the list - null terminated!- of known pots
 */

char **
definedPots (char kind)
{
  char **list = NULL;

  for (int i = 0; i < knownPots.n; i++)
    if (kind == knownPots.kind[i])
      list = nllAddDup (list, knownPots.name[i]);

  return list;
}

/*
 * @parameter kind, potential type, '2' or '3'
 * @return the potential's internal number
 */
int
getPotNumber (char kind)
{
  int ret = -1;
  switch (kind)
    {
    case '2':
      ret = ActualPot;
      break;
    case '3':
      ret = Actual3Pot;
      break;
    }

  return ret;
}

/*
 * @parameters kind interaction type, '2' or '3'
 * @return the number of coefficients.
 */
int
getPotCoefs (char kind)
{
  int ret = -1;
  switch (kind)
    {
    case '2':
      ret = knownPots.ncoefs[ActualPot];
      break;
    case '3':
      ret = knownPots.ncoefs[Actual3Pot];
      break;
    }
  if (ret < 0)
    fcStopIt ("Potential not set");
  return ret;
}

/*
 * @parameter kind interaction type, '2' or '3'
 * @return potential coding language.
 */
char
getPotLang (char kind)
{
  int ret = -1;
  switch (kind)
    {
    case '2':
      ret = knownPots.lang[ActualPot];
      break;
    case '3':
      ret = knownPots.lang[Actual3Pot];
      break;
    }
  return ret;
}


/*
 * @parameter kind interaction type, '2' or '3'
 * @return fortran function
 */
int
getPotFunctionF (char kind)
{
  int pot = (kind == '2' ? ActualPot : (kind == '3' ? Actual3Pot : -1));
  if (pot < 0)
    fcStopIt ("Fortran potential not set");

  if (knownPots.lang[pot] == 'f')
    return knownPots.npot[pot];
  return -1;
}


/*
 * @parameter kind interaction type, '2' or '3'
 * @return C function
 */
FPOINTER
getPotFunctionC (char kind)
{
  int pot = (kind == '2' ? ActualPot : (kind == '3' ? Actual3Pot : -1));
  if (pot < 0)
    fcStopIt ("C potential not set");

  if (knownPots.lang[pot] == 'c')
    return knownPots.funcs[pot];
  return NULL;
}


/*
 * @parameter kind interaction type, '2' or '3'
 * @parameter name potential name
 */
void
setPotential (char kind, char *name)
{
  for (int i = 0; i < knownPots.n; i++)
    {
      if (strcmp (knownPots.name[i], name) == 0)
	{
	  if (kind == '2')
	    ActualPot = i;
	  if (kind == '3')
	    Actual3Pot = i;
	  return;
	}
    }
  fcPrintfStopIt ("Potential %s unknown\n", name);
}

/*
 * @parameter msg message
 * @parameter n message length
 * It's called from fortran
 */
void
FCStopIt (char *msg, long n)
{
  char s[STRING_MAX];
  strncpy (s, msg, n);
  fcStopIt (msg);
}
