/*
(c) GAFit toolkit $Id: multijob.c 562 2025-12-11 15:34:12Z ro $
*/
#if HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <float.h>
#include <ctype.h>
#include "../core.h"
#include "../nullist/nllist.h"
#include "../flyctl/flyctl.h"
#include "../inputline/line.h"
#include "../cnames.h"
#include "../numbers.h"
#include "../parameters/parameters.h"
#include "../rstrings/rstrings.h"
#include "../job.h"
#include "regpots.h"
#include "multijob.h"
#include "multi.h"
#include "auxfuncs.h"
#include "bounds.h"
#include "3body.h"
#include "hbonds.h"

void pird (char *format, ...);
void dmp (char *mark);

int dg = DBL_DECIMAL_DIG;


/*
 * To manage a upper triangular matrix
 * @param row the row
 * @param col the column
 * @return the correct row
 */
int
srow (int row, int col)
{
  if (row < col)
    return row;
  return col;
}

/*
 * To maname a upper triangular matrix
 * @param row the row
 * @param col the col
 * @return the correct column
 */
int
scol (int row, int col)
{
  if (row < col)
    return col;
  return row;
}

/*
 * Calculates the index in the interaction vector
 * which represent the upper triangular matrix
 * from row and column
 * @param j the multijob 
 * @param ro the row
 * @param co the column
 * @return the index
 */
int
mat2vecBase1 (MULTIJOB * j, int ro, int co)
{
  int r;
  int row = srow (ro, co);
  int col = scol (ro, co);
  r = (2 * j->dtypes + 1 - (row - 1)) * (row - 1) / 2 + col - row + 1;
  return r;
}

/*
 * interaction vector length
 * @param j the multijob
 * @return the vector length
 * */
int
vecLen (MULTIJOB * j)
{
  int len = mat2vecBase1 (j, j->dtypes, j->dtypes);
  return len;
}

/*
 * (row,col)->vector index
 * @param j multijob
 * @param ro row
 * @para co column
 * @return vector index.
**/
int
mat2vec (MULTIJOB * j, int ro, int co)
{
  return mat2vecBase1 (j, ro + 1, co + 1) - 1;
}


void
dimGeomG (MULTIJOB * j, int sys, int ne)
{
  //ne different geometries for system  'sys'
  j->data[sys]->geos = (GEOMETRY **) malloc (sizeof (GEOMETRY *) * ne);
  for (int k = 0; k < j->data[sys]->ne; k++)
    {
      j->data[sys]->geos[k] = malloc (sizeof (GEOMETRY));
      j->data[sys]->geos[k]->atom =
	(char **) malloc (sizeof (char *) * j->data[sys]->n);
      j->data[sys]->geos[k]->x =
	(double *) malloc (sizeof (double) * j->data[sys]->n);
      j->data[sys]->geos[k]->y =
	(double *) malloc (sizeof (double) * j->data[sys]->n);
      j->data[sys]->geos[k]->z =
	(double *) malloc (sizeof (double *) * j->data[sys]->n);
    }
}

void
dimGeom (MULTIJOB * j)
{
  // j->nsys different systems
  for (int i = 0; i < j->nsys; i++)
    {
      dimGeomG (j, i, j->data[i]->ne);
      if (j->debug)
	pird ("s.%d geos %d\n", "dim geo", i, j->data[i]->ne);
    }
}

void
dimA2tntype (MULTIJOB * j, int system)
{
  j->data[system]->a2tntype =
    (int *) malloc (sizeof (int) * j->data[system]->n);
}

/**
 * builds the right upper triangle matrix in a lineal
 * vector.
 * @param j the MULTIJOB afected.
 * */
void
dimInterVector (MULTIJOB * j)
{
  int len = vecLen (j);
  j->count2body = 0;
  j->interactions = (int *) malloc (sizeof (int) * len);
  for (int i = 0; i < len; i++)
    j->interactions[i] = 0;
}

void
dimData (MULTIJOB * j)
{
  j->data = malloc (sizeof (SYSTEM *) * j->nsys);
  for (int i = 0; i < j->nsys; i++)
    {
      j->data[i] = malloc (sizeof (SYSTEM));
      j->data[i]->frag = 0;
      j->data[i]->n = 0;
      j->data[i]->a2tatom = NULL;
      j->data[i]->a2ttype = NULL;
      j->data[i]->a2tntype = NULL;
      j->data[i]->ne = 0;
      j->data[i]->energies = NULL;
      j->data[i]->weigth = NULL;
      j->data[i]->dtypea = NULL;
      j->data[i]->dtypeb = NULL;
    };
}

int
type2num (char *type, MULTIJOB * j)
{
  int res = nllLocate (type, j->types);
  if (res < 0)
    fcPrintfStopIt ("Error type %s not found\n", type);
  return res;
}

void
lPrint (char **l)
{
  char first = 1;
  char **p = l;
  while (*p)
    {
      if (!first)
	printf (", ");
      printf ("%s", *p);
      p++;
      first = 0;
    }
}

void
inter2vector (MULTIJOB * j, char **l1, char **l2)
{
  int a, b;
  int n1 = nllCount (l1), n2 = nllCount (l2);

  for (int i = 0; i < n1; i++)
    {
      a = type2num (l1[i], j);
      for (int k = 0; k < n2; k++)
	{
	  b = type2num (l2[k], j);
	  int index = mat2vec (j, a, b);
	  if (j->interactions[index] == 0)
	    {
	      ++j->count2body;
	    }
	  j->interactions[index]++;
	}
    }
}

void
renumInterVector (MULTIJOB * j)
{
  int n = 0;

  for (int i = 0; i < j->dtypes; i++)
    for (int k = i; k < j->dtypes; k++)
      if (j->interactions[mat2vec (j, i, k)] != 0)
	j->interactions[mat2vec (j, i, k)] = ++n;
}

/*
 * create a new multijob
 * @return the new multijob struct.
 */
MULTIJOB *
nwmj (void)
{
  MULTIJOB *mj = malloc (sizeof (MULTIJOB));
  mj->debug = 0;
  mj->msummary = 0;
  mj->potential = 0;
  mj->potential_name = NULL;
  mj->fitting = 0;
  mj->rcoefs = 0;
  mj->nsys = 0;
  mj->systems = NULL;
  mj->geometries = NULL;
  mj->energies = NULL;
  mj->atom2type = NULL;
  mj->data = NULL;
  mj->dtypes = 0;
  mj->types = NULL;
  mj->count2body = 0;
  mj->interactions = NULL;

  mj->tbinteractions = 0;
  mj->tbody = NULL;

  mj->hbinteractions = 0;
  mj->hbonds = NULL;

  mj->job = (JOB *) malloc (sizeof (JOB));

  mj->job->file = strdup (INPUT_FILE_INTJOB);
  mj->job->external_input = (char *) malloc (sizeof (char) * STRING_MAX);
  mj->job->external_fit = (char *) malloc (sizeof (char) * STRING_MAX);
  mj->job->external_bounds = (char *) malloc (sizeof (char) * STRING_MAX);

  mj->vec_coeffs = NULL;
  return mj;
}


void
saveData (MULTIJOB * j, char *file)
{
  char *tfit[] = LIST_FITS;
  char *hbkinds[] = LIST_HB_KINDS;

  FILE *f = fopen (file, "w");
  int n = 0;
  int ta, tb;
  int vlen = vecLen (j);

  setMPointer (j);

  char *fmt_geo = "%4d %4d %4d %4d %4s %a %a %a\n";
  char *fmt_geo_a = "%4d %4d %4d %4d %4s %g %g %g\n";
  char *fmt_energies = "%4d %4d %4d %a %a\n";
  char *fmt_energies_a = "%4d %4d %4d %g %g\n";
  if (j->debug)
    {
      fmt_geo = fmt_geo_a;
      fmt_energies = fmt_energies_a;
    }

  fprintf (f, "**GAFIT MULTI MODULE DATA**\n");
  fprintf (f, "%4d %4s %d\n", n, "debug", j->debug);
  fprintf (f, "%4d %4s %s\n", ++n, "potential", j->potential_name);
  fprintf (f, "%4d %4s %s\n", ++n, "fitting", tfit[j->fitting]);
  fprintf (f, "%4d %4s %d\n", ++n, "msummary", j->msummary);
  fprintf (f, "%4d %4s %d\n", ++n, "nsys", j->nsys);
  for (int i = 0; i < j->nsys; i++)
    {
      fprintf (f, "%4d %4d %s\n", ++n, i + 1, j->systems[i]);
      fprintf (f, "%4d %4d %s\n", ++n, i + 1, j->atom2type[i]);
      fprintf (f, "%4d %4d %s\n", ++n, i + 1, j->energies[i]);
      fprintf (f, "%4d %4d %s\n", ++n, i + 1, j->geometries[i]);
      fprintf (f, "%4d %4d %4s %d\n", ++n, i + 1, "natoms", j->data[i]->n);
      fprintf (f, "%4d %4d %4s %d\n", ++n, i + 1, "frag", j->data[i]->frag);

      for (int k = 0; k < j->data[i]->n; k++)
	{
	  fprintf (f, "%4d %4d %4d %4s %4s %4d\n", ++n, i + 1, k + 1,
		   j->data[i]->a2tatom[k], j->data[i]->a2ttype[k],
		   j->data[i]->a2tntype[k] + 1);
	}
      fprintf (f, "%4d %4d %4s %d\n", ++n, i + 1, "ntypea",
	       j->data[i]->ntypea);
      for (int k = 0; k < j->data[i]->ntypea; k++)
	{
	  fprintf (f, "%4d %4d %4d %4s\n", ++n, i + 1, k + 1,
		   j->data[i]->dtypea[k]);
	}
      fprintf (f, "%4d %4d %4s %d\n", ++n, i + 1, "ntypeb",
	       j->data[i]->ntypeb);

      for (int k = 0; k < j->data[i]->ntypeb; k++)
	{
	  fprintf (f, "%4d %4d %4d %4s\n", ++n, i + 1, k + 1,
		   j->data[i]->dtypeb[k]);
	}

      fprintf (f, "%4d %4d %4s %d\n", ++n, i + 1, "ne", j->data[i]->ne);

      for (int k = 0; k < j->data[i]->ne; k++)
	{
	  fprintf (f, fmt_energies, ++n, i + 1, k + 1,
		   j->data[i]->energies[k], j->data[i]->weigth[k]);
	  for (int l = 0; l < j->data[i]->n; l++)
	    {
	      fprintf (f, fmt_geo,
		       ++n, i + 1, k + 1, l + 1, j->data[i]->geos[k]->atom[l],
		       j->data[i]->geos[k]->x[l], j->data[i]->geos[k]->y[l],
		       j->data[i]->geos[k]->z[l]);
	    }
	}
    }

  fprintf (f, "%4d %4s %d\n", ++n, "dtypes", j->dtypes);
  for (int i = 0; i < j->dtypes; i++)
    {
      fprintf (f, "%4d %4d %4s\n", ++n, i + 1, j->types[i]);
    }

  fprintf (f, "%4d %4s %d\n", ++n, "count2body", j->count2body);


  j->dbodyint = (DBODYINT *) malloc (sizeof (DBODYINT) * j->count2body);

  for (int i = 0; i < vlen; i++)
    {

      if (j->interactions[i] != 0)
	{
	  strInterAtomType (j->interactions[i], &ta, &tb);
	  //-> hook 2body str interactions

	  j->dbodyint[j->interactions[i] - 1].str =
	    strdup (strInterType (j->interactions[i]));

	  j->dbodyint[j->interactions[i] - 1].typea = ta;

	  j->dbodyint[j->interactions[i] - 1].typeb = tb;
	  fprintf (f, "%4d %4d %4d %4d %4d %4s %4s\n", ++n, i + 1,
		   j->interactions[i], ta, tb, "type",
		   strInterType (j->interactions[i]));
	}
      else
	{
	  // a hole in the matrix. A interaction NON existent in the systems
	  // j->inteeractions[i]==0
	  fprintf (f, "%4d %4d %4d %4d %4d %4s %4s\n", ++n, i + 1,
		   j->interactions[i], 0, 0, "type", "NONE");
	}
    }

  if (j->tbinteractions)
    {
      fprintf (f, "%4d %4s %d\n", ++n, "3body", j->tbinteractions);
      fprintf (f, "%4d %4s %4s\n", ++n, "potential",
	       j->tbody->potential_name);
      fprintf (f, "%4d %4s %d\n", ++n, "count3body", j->tbody->count3body);
      for (int i = 0; i < j->tbody->count3body; i++)
	{
	  fprintf (f, "%4d %4d %4d %4d %4d %4s %4s\n", ++n, i + 1,
		   j->tbody->tbodyint[i].typea + 1,
		   j->tbody->tbodyint[i].typeb + 1,
		   j->tbody->tbodyint[i].typec + 1, "type",
		   j->tbody->tbodyint[i].str);
	}
    }
  else
    {
      fprintf (f, "%4d %4s %d\n", ++n, "3body", 0);
    }

  fprintf (f, "%4d %s %d\n", ++n, "hbonds", j->hbinteractions);

  if (j->hbinteractions)
    {
      fprintf (f, "%4d %s %d\n", ++n, "countHb", j->hbonds->countHb);
      fprintf (f, "%4d %s %d\n", ++n, "ncoefs", j->hbonds->ncoefs);
      fprintf (f, "%4d %s %s\n", ++n, "kind", hbkinds[j->hbonds->kind]);
      fprintf (f, "%4d %s %d\n", ++n, "debug", j->hbonds->debug);

      for (int i = 0; i < j->hbonds->ncoefs; i++)
	{
	  fprintf (f, "%4d %4d %s %s\n", ++n, i + 1, "coefficient",
		   j->hbonds->hb_groups[i]->hbg_name);
	  fprintf (f, "%4d %4d %s %d\n", ++n, i + 1, "n_types",
		   j->hbonds->hb_groups[i]->n_types);

	  for (int k = 0; k < j->hbonds->hb_groups[i]->n_types; k++)
	    {
	      fprintf (f, "%4d %4d %4d %s %s\n", ++n, i + 1, k + 1, "a2type",
		       nllString (j->hbonds->hb_groups[i]->a_types, k));
	    }
	}

      for (int i = 0; i < j->nsys; i++)
	{
	  for (int k = 0; k < j->data[i]->n; k++)
	    {
	      fprintf (f, "%4d %4d %4d %2s %8s %4d %4d %s\n", ++n, i + 1,
		       k + 1, j->hbonds->hbsys[i]->atoms[k],
		       j->hbonds->hbsys[i]->types[k],
		       j->hbonds->hbsys[i]->type_n[k] + 1,
		       j->hbonds->hbsys[i]->hb_coef[k],
		       getHbCoefString (j->hbonds->hbsys[i]->hb_coef[k],
					j->hbonds));
	    }
	}
    }

  fprintf (f, "%4d %s %s\n", ++n, DATA_END, DATA_DATA);

  fclose (f);
}


/** 
 * total number of coefficients
 */
int
NC (MULTIJOB * j)
{
  int total = j->job->N;
  if (j->tbinteractions)
    total += j->tbody->N;
  if (j->hbinteractions)
    total += j->hbonds->N;
  return total;
}

void
loadData (MULTIJOB * j, char *file)
{
  FILE *f = fopen (file, "r");
  int n, m, o, p, q;
  char s[STRING_MAX];
  char z[STRING_MAX];
  char y[STRING_MAX];

  char *fmt_geo = "%d %d %d %d %s %la %la %la\n";
  char *fmt_geo_a = "%d %d %d %d %s %lg %lg %lg\n";
  char *fmt_energies = "%d %d %d %la %la\n";
  char *fmt_energies_a = "%d %d %d %lg %lg\n";

  char *tfit[] = LIST_FITS;
  char *hbkind[] = LIST_HB_KINDS;

  int vlen;

  int ncoefs2plusx = 0;

  if (!f)
    fcPrintfStopIt ("I cannot read file %s\n", file);

  fscanf (f, "%s %s %s %s", s, s, s, s);	//reads banner

  fscanf (f, "%d %s %d", &n, s, &m);

//entry point after debug flag

  j->debug = m;			//reads debug flag

  if (j->debug)
    {
      fmt_geo = fmt_geo_a;
      fmt_energies = fmt_energies_a;
    }

  fscanf (f, "%d %s %s", &n, s, z);

  j->potential_name = strdup (z);

  setPotential ('2', j->potential_name);

  j->job->ncoefs = getPotCoefs ('2');

  ncoefs2plusx = j->job->ncoefs;

  //setup coefficient's names.
  //cnFile (j->job->file, j->job->ncoefs);

  j->potential = getPotNumber ('2');

  fscanf (f, "%d %s %s", &n, s, z);
  j->fitting = ParameterSet2int (z, tfit);

  fscanf (f, "%d %s %d", &n, s, &m);
  j->msummary = m;

  fscanf (f, "%d %s %d", &n, s, &j->nsys);

  dimData (j);

  for (int i = 0; i < j->nsys; i++)
    {
      fscanf (f, "%d %d %s", &n, &m, z);
      j->systems = nllAddDup (j->systems, z);

      fscanf (f, "%d %d %s", &n, &m, z);
      j->atom2type = nllAddDup (j->atom2type, z);

      fscanf (f, "%d %d %s", &n, &m, z);
      j->energies = nllAddDup (j->energies, z);
      fscanf (f, "%d %d %s", &n, &m, z);
      j->geometries = nllAddDup (j->geometries, z);

      fscanf (f, "%d %d %s %d", &n, &m, s, &j->data[i]->n);
      fscanf (f, "%d %d %s %d", &n, &m, s, &j->data[i]->frag);

      dimA2tntype (j, i);
      for (int k = 0; k < j->data[i]->n; k++)
	{
	  int p;
	  fscanf (f, "%d %d %d %s %s %d", &n, &m, &o, s, z, &p);
	  j->data[i]->a2tatom = nllAddDup (j->data[i]->a2tatom, s);
	  j->data[i]->a2ttype = nllAddDup (j->data[i]->a2ttype, z);
	  j->data[i]->a2tntype[k] = p - 1;
	}

      fscanf (f, "%d %d %s %d", &n, &m, s, &j->data[i]->ntypea);

      for (int k = 0; k < j->data[i]->ntypea; k++)
	{
	  fscanf (f, "%d %d %d %s", &n, &m, &o, s);
	  j->data[i]->dtypea = nllAddDup (j->data[i]->dtypea, s);
	}

      fscanf (f, "%d %d %s %d", &n, &m, s, &j->data[i]->ntypeb);

      for (int k = 0; k < j->data[i]->ntypeb; k++)
	{
	  fscanf (f, "%d %d %d %s", &n, &m, &o, s);
	  j->data[i]->dtypeb = nllAddDup (j->data[i]->dtypeb, s);
	}

      fscanf (f, "%d %d %s %d", &n, &m, s, &j->data[i]->ne);

      dimGeomG (j, i, j->data[i]->ne);

      j->data[i]->energies =
	(double *) malloc (sizeof (double) * j->data[i]->ne);
      j->data[i]->weigth =
	(double *) malloc (sizeof (double) * j->data[i]->ne);


      for (int k = 0; k < j->data[i]->ne; k++)
	{
	  fscanf (f, fmt_energies, &n, &m, &o,
		  &j->data[i]->energies[k], &j->data[i]->weigth[k]);

	  for (int l = 0; l < j->data[i]->n; l++)
	    {

	      fscanf (f, fmt_geo, &n, &m, &o, &p,
		      s,
		      &j->data[i]->geos[k]->x[l], &j->data[i]->geos[k]->y[l],
		      &j->data[i]->geos[k]->z[l]);

	      j->data[i]->geos[k]->atom[l] = strdup (s);
	    }
	}
    }

  fscanf (f, "%d %s %d\n", &n, s, &j->dtypes);

  dimInterVector (j);

  vlen = vecLen (j);

  for (int i = 0; i < j->dtypes; i++)
    {
      fscanf (f, "%d %d %s", &n, &m, s);
      j->types = nllAddDup (j->types, s);
    }


  fscanf (f, "%d %s %d", &n, s, &j->count2body);

  j->job->N = j->job->ncoefs * j->count2body;

  j->dbodyint = (DBODYINT *) calloc (1, sizeof (DBODYINT) * vlen);

  for (int i = 0; i < vlen; i++)
    {

      fscanf (f, "%d %d %d %d %d %s %s\n", &n, &m, &j->interactions[i], &p,
	      &q, z, s);

      if (j->interactions[i] != 0)
	{
	  j->dbodyint[j->interactions[i] - 1].typea = p - 1;
	  j->dbodyint[j->interactions[i] - 1].typeb = q - 1;
	  j->dbodyint[j->interactions[i] - 1].str = strdup (s);
	}
      else
	{
	  // a interaction not existent (type NONE), a hole in the matrix
	}
    }

/// 3body data ///////////////////////////////////////////////////////////////
  fscanf (f, "%d %s %d\n", &n, s, &j->tbinteractions);
  if (j->tbinteractions)
    {
      j->tbody = malloc (sizeof (TBODY));

      fscanf (f, "%d %s %s", &n, s, z);

      j->tbody->potential_name = strdup (z);

      setPotential ('3', j->tbody->potential_name);

      j->tbody->ncoefs = getPotCoefs ('3');

      ncoefs2plusx += j->tbody->ncoefs;

      j->tbody->potential = getPotNumber ('3');

      fscanf (f, "%d %s %d\n", &n, s, &j->tbody->count3body);

      j->tbody->N = j->tbody->count3body * j->tbody->ncoefs;

      j->tbody->tbodyint = malloc (sizeof (TBODYINT) * j->tbody->count3body);

      j->tbody->type2intCube = (int ***) malloc (sizeof (int **) * j->dtypes);

      for (int i = 0; i < j->dtypes; i++)
	{
	  j->tbody->type2intCube[i] =
	    (int **) malloc (sizeof (int *) * j->dtypes);
	  for (int k = 0; k < j->dtypes; k++)
	    {
	      j->tbody->type2intCube[i][k] =
		(int *) malloc (sizeof (int) * j->dtypes);
	    }
	}

      for (int i = 0; i < j->tbody->count3body; i++)
	{
	  fscanf (f, "%d %d %d %d %d %s %s\n", &n, &m, &o, &p, &q, s, z);
	  j->tbody->tbodyint[i].typea = o - 1;
	  j->tbody->tbodyint[i].typeb = p - 1;
	  j->tbody->tbodyint[i].typec = q - 1;
	  j->tbody->tbodyint[i].str = strdup (z);
	  j->tbody->type2intCube[o - 1][p - 1][q - 1] = i + 1;	//interactions from 1
	}
    }

/// hb3  data ///////////////////////////////////////////////////////////////
  fscanf (f, "%d %s %d\n", &n, s, &j->hbinteractions);
  if (j->hbinteractions)
    {
      j->hbonds = malloc (sizeof (HBONDS));
      fscanf (f, "%d %s %d\n", &n, s, &j->hbonds->countHb);
      fscanf (f, "%d %s %d\n", &n, s, &j->hbonds->ncoefs);

      fscanf (f, "%d %s %s\n", &n, s, z);
      j->hbonds->kind = ParameterSet2int (z, hbkind);

      fscanf (f, "%d %s %d\n", &n, s, &j->hbonds->debug);

      j->hbonds->hb_groups = malloc (sizeof (HBGROUP *) * j->hbonds->ncoefs);

      for (int i = 0; i < j->hbonds->ncoefs; i++)
	{
	  j->hbonds->hb_groups[i] = malloc (sizeof (HBGROUP));

	  fscanf (f, "%d %d %s %s\n", &n, &m, s, z);
	  j->hbonds->hb_groups[i]->hbg_name = strdup (z);

	  fscanf (f, "%d %d %s %d\n", &n, &m, s, &o);
	  j->hbonds->hb_groups[i]->n_types = o;
	  j->hbonds->hb_groups[i]->a_types = NULL;

	  for (int k = 0; k < j->hbonds->hb_groups[i]->n_types; k++)
	    {
	      fscanf (f, "%d %d %d %s %s\n", &n, &m, &o, s, z);
	      j->hbonds->hb_groups[i]->a_types =
		nllAddDup (j->hbonds->hb_groups[i]->a_types, z);
//            printf ("%d %d %d %s %s\n", n, m, o, s, z);
	    }
	}
      j->hbonds->nsys = j->nsys;
      j->hbonds->N = j->hbonds->ncoefs * 1;	// one interaction

      j->hbonds->hbsys = malloc (sizeof (HBSYSTEMS *) * j->nsys);

      for (int i = 0; i < j->nsys; i++)
	{
	  j->hbonds->hbsys[i] = malloc (sizeof (HBSYSTEMS));
	  j->hbonds->hbsys[i]->n_atoms = j->data[i]->n;
	  j->hbonds->hbsys[i]->frag = j->data[i]->frag;
	  j->hbonds->hbsys[i]->atoms =
	    malloc (sizeof (char *) * j->data[i]->n);
	  j->hbonds->hbsys[i]->types =
	    malloc (sizeof (char *) * j->data[i]->n);
	  j->hbonds->hbsys[i]->type_n = malloc (sizeof (int) * j->data[i]->n);
	  j->hbonds->hbsys[i]->hb_coef =
	    malloc (sizeof (int) * j->data[i]->n);

	  for (int k = 0; k < j->data[i]->n; k++)
	    {
	      fscanf (f, "%d %d %d %s %s %d %d %s\n", &n, &m, &o, s, z, &p,
		      &q, y);
	      j->hbonds->hbsys[i]->atoms[k] = strdup (s);
	      j->hbonds->hbsys[i]->types[k] = strdup (z);
	      j->hbonds->hbsys[i]->type_n[k] = p - 1;
	      j->hbonds->hbsys[i]->hb_coef[k] = q;	//not q-1!!! this is passed to fortran
	      // also getHbCoefString.
//            printf("%d %d %d %s %s %d %d %s\n",n,m,o,s,z,p,q,getHbCoefString(q,j->hbonds));
	    }

	}
// memory block to pass labels to fortran as an array not null terminated
// and geometries adapted 
      for (int i = 0; i < j->hbonds->nsys; i++)
	{
	  j->hbonds->hbsys[i]->labels_block =
	    malloc (sizeof (char) * 2 * j->hbonds->hbsys[i]->n_atoms);
	  j->hbonds->hbsys[i]->ilabels_block =
	    malloc (sizeof (int) * j->hbonds->hbsys[i]->n_atoms);
	  for (int k = 0; k < j->hbonds->hbsys[i]->n_atoms; k++)
	    {
	      strncpy (&j->hbonds->hbsys[i]->labels_block[k * 2],
		       j->hbonds->hbsys[i]->atoms[k], 2);
	    }
	  for (int k = 0; k < j->hbonds->hbsys[i]->n_atoms * 2; k++)
	    {
	      j->hbonds->hbsys[i]->labels_block[k] =
		tolower (j->hbonds->hbsys[i]->labels_block[k]);
	    }
	  for (int k = 0; k < j->hbonds->hbsys[i]->n_atoms; k++)
	    {
	      if ((j->hbonds->hbsys[i]->ilabels_block[k] =
		   atomicNumber (j->hbonds->hbsys[i]->atoms[k])) == 0)
		{
		  fcPrintfStopIt ("I cannot assign atomic number to %s\n",
				  j->hbonds->hbsys[i]->atoms[k]);
		}
	    }
	  // geos ----------------------------------------->>>>>>
	  j->hbonds->hbsys[i]->ne = j->data[i]->ne;
	  j->hbonds->hbsys[i]->geo =
	    malloc (sizeof (double *) * j->hbonds->hbsys[i]->ne);

	  // building ne'th fortran like 2D matrix
	  for (int k = 0; k < j->hbonds->hbsys[i]->ne; k++)
	    {
	      double *wgeo =
		malloc (sizeof (double) * 3 * j->hbonds->hbsys[i]->n_atoms);
	      for (int m = 0; m < j->hbonds->hbsys[i]->n_atoms; m++)
		{
		  wgeo[m * 3] = j->data[i]->geos[k]->x[m] / BOHR2ANGSTROM;
		  wgeo[m * 3 + 1] = j->data[i]->geos[k]->y[m] / BOHR2ANGSTROM;
		  wgeo[m * 3 + 2] = j->data[i]->geos[k]->z[m] / BOHR2ANGSTROM;
		}
	      j->hbonds->hbsys[i]->geo[k] = wgeo;
	    }
	}
    }

  fscanf (f, "%d %s %s", &n, s, z);
  if (strcmp (s, DATA_END) || strcmp (z, DATA_DATA))
    {
      printf ("error lectura!");
      printf ("%d [%s]-[%s]/[%s]-[%s]\n", n, s, DATA_END, z, DATA_DATA);
    }
  fclose (f);
/// coefficient's names
//cnFile (j->job->file, NC (j) - j->hbonds->N);
//if (j->hbinteractions)
//  {
//    char **names = getHbCoefStringS (j->hbonds);
//    cnAdd (names);
//  }

  setCNames (j, ncoefs2plusx);
//ncoefs2plus3plusHb += j->hbonds->ncoefs;
  if (j->debug)
    {
// if debug save read values
      saveData (j, DUMP_DATA_FILE);
    }
}

void
addA2t (MULTIJOB * j, int system, char *atom, char *type)
{
  j->data[system]->a2tatom = nllAddDup (j->data[system]->a2tatom, atom);
  j->data[system]->a2ttype = nllAddDup (j->data[system]->a2ttype, type);
}

void
pBar (MULTIJOB * j, int num)
{
  char fb[] = "             ";
  if (num)
    {
      printf (fb);
      for (int i = 0; i < j->dtypes; i++)
	printf ("  %4d  ", i + 1);
      printf ("\n");
      printf (fb);
      for (int i = 0; i < j->dtypes; i++)
	printf ("  %4s  ", j->types[i]);
      printf ("\n");
    }
  printf ("-----------+-");
  for (int i = 0; i < j->dtypes; i++)
    printf ("--------");
  printf ("+\n");
}

void
printInteractions (MULTIJOB * j)
{
  printf ("\n");
  printf ("INTERACTIONS MATRIX\n");
  printSepBar (stdout);
  pBar (j, 1);
  for (int i = 0; i < j->dtypes; i++)
    {
      printf (" %4d %4s | ", i + 1, j->types[i]);
      for (int k = 0; k < i; k++)
	printf ("        ");
      for (int k = i; k < j->dtypes; k++)
	{
	  if (j->interactions[mat2vec (j, i, k)] != 0)
	    printf ("[%4d]  ", j->interactions[mat2vec (j, i, k)]);
	  else
	    printf ("[----]  ");
	}
      printf ("|\n");
    }

  pBar (j, 0);
}

void
printTypes (MULTIJOB * j)
{
  printf ("Found %d systems with %d distinct atom types: ",
	  j->nsys, j->dtypes);
  lPrint (j->types);
  printf ("\n");
  for (int i = 0; i < j->nsys; i++)
    {
      printf ("System %s, ", j->systems[i]);
      printf ("\n\t- ( ");
      lPrint (j->data[i]->dtypea);
      printf (" ) x ( ");
      lPrint (j->data[i]->dtypeb);
      printf (" ): %d interactions",
	      nllCount (j->data[i]->dtypea) * nllCount (j->data[i]->dtypeb));
      printf ("\n");
      printf ("\t- %d geometries\n", j->data[i]->ne);
    }

  printf ("\n");
  printf ("Overall different interactions: %d\n", j->count2body);
  printf ("\tWe need a %d coefficients vector.\n", j->job->N);
  printf ("\n");
}

void
readAtom2Type (MULTIJOB * j)
{
  FILE *f;
  char *s;
  int a, t, n;
  char atom[STRING_MAX], type[STRING_MAX];
//char tmp[STRING_MAX];
  char **l1, **l2;
  for (int i = 0; i < j->nsys; i++)
    {
      if (!FileExist (j->atom2type[i]))
	fcPrintfStopIt ("File doesn't exist:%s\n", j->atom2type[i]);
      f = fopen (j->atom2type[i], "r");
      s = InputFileLine (f);
      sscanf (s, "%d %d", &a, &t);
      j->data[i]->frag = a;
      j->data[i]->n = t;
      while ((s = InputFileLine (f)) != NULL)
	{
	  sscanf (s, "%d %s %s", &n, atom, type);
	  addA2t (j, i, atom, type);
	  free (s);
	}
      fclose (f);
      if (!(n == t))
	fcPrintfStopIt ("atom2types %s %d <> %d\n", n, t, j->atom2type[i]);
    }

  l1 = nllListDup (j->data[0]->a2ttype);
  for (int i = 1; i < j->nsys; i++)
    {
      l2 = nllListUnion (l1, j->data[i]->a2ttype);
      nllClear (l1);
      l1 = l2;
    }

  j->dtypes = nllCount (l1);
  j->types = l1;
  for (int i = 0; i < j->nsys; i++)
    {
      dimA2tntype (j, i);
      for (int k = 0; k < j->data[i]->n; k++)
	{
	  j->data[i]->a2tntype[k] = type2num (j->data[i]->a2ttype[k], j);
	}
    }

  dimInterVector (j);
  for (int i = 0; i < j->nsys; i++)
    {
      l1 = l2 = NULL;
      l1 = nllSubList (j->data[i]->a2ttype, 1, j->data[i]->frag);
      l2 =
	nllSubList (j->data[i]->a2ttype, j->data[i]->frag + 1, j->data[i]->n);
      l1 = nllUnic (l1);
      l2 = nllUnic (l2);
      inter2vector (j, l1, l2);
      j->data[i]->dtypea = l1;
      j->data[i]->ntypea = nllCount (l1);
      j->data[i]->dtypeb = l2;
      j->data[i]->ntypeb = nllCount (l2);
    }

  renumInterVector (j);
}

void
addEnergy (MULTIJOB * j, int system, double ener, double wei)
{
  int k;
  k = ++j->data[system]->ne;
  j->data[system]->energies = (double *)
    realloc (j->data[system]->energies, sizeof (double) * k);
  j->data[system]->energies[k - 1] = ener;
  j->data[system]->weigth = (double *)
    realloc (j->data[system]->weigth, sizeof (double) * k);
  j->data[system]->weigth[k - 1] = wei;
}

void
printEnergies (MULTIJOB * j)
{
  for (int i = 0; i < j->nsys; i++)
    {
      printf ("Energies: %s,  %d values\n", j->energies[i], j->data[i]->ne);
      for (int k = 0; k < j->data[i]->ne; k++)
	printf (" %4d %.*g %.*g\n", k + 1, dg,
		j->data[i]->energies[k], dg, j->data[i]->weigth[k]);
    }
}

void
readEnergies (MULTIJOB * j)
{
  FILE *f;
  char *s;
  int r;
  double energy, weigth;
  int count = 0;
  for (int i = 0; i < j->nsys; i++)
    {
      if (!FileExist (j->energies[i]))
	fcPrintfStopIt ("File doesn't exist:%s\n", j->energies[i]);
      f = fopen (j->energies[i], "r");
      while ((s = InputFileLine (f)) != NULL)
	{
	  r = sscanf (s, "%lf %lf", &energy, &weigth);
	  if (r == 2)
	    {
	      addEnergy (j, i, energy, weigth);
	      count++;
	    }
	  else
	    {
	      if (j->debug)
		pird ("not read: %s\n", "energies", s);
	    }
	  free (s);
	}
      if (j->debug)
	pird ("s.%d ener %d\n", "energies", i, count);
      count = 0;
      fclose (f);
    }
  return;
}

void
addGeometry (MULTIJOB * j, int system, int geo,
	     int line, char *atom, double x, double y, double z)
{
  j->data[system]->geos[geo]->atom[line] = strdup (atom);
  j->data[system]->geos[geo]->x[line] = x;
  j->data[system]->geos[geo]->y[line] = y;
  j->data[system]->geos[geo]->z[line] = z;
}

void
readGeometries (MULTIJOB * j)
{
  FILE *f;
  char *s;
  int r;
  int natom;
  int ngeos;
  double x, y, z;
  char atom[STRING_MAX];
  int line;
  dimGeom (j);
  for (int i = 0; i < j->nsys; i++)
    {
      if (j->debug)
	{
	  pird ("s.%d #atom %d\n", "system", i, j->data[i]->n);
	}

      if (!FileExist (j->geometries[i]))
	fcPrintfStopIt ("File doesn't exist:%s\n", j->geometries[i]);
      f = fopen (j->geometries[i], "r");
      ngeos = 0;
      natom = 0;
      line = 0;
      while ((s = InputFileLine (f)) != NULL)
	{
	  r = sscanf (s, "%d", &natom);
	  line++;
	  if (natom != j->data[i]->n)
	    {
	      fcPrintfStopIt
		("Geometries: system %s, geo file %s, geo #%d, read %d atoms, not %d\nline:%d '%s'",
		 j->systems[i], j->geometries[i], ngeos, natom,
		 j->data[i]->n, line, s);
	    }
	  else
	    {
	      if (j->debug)
		{
		  printf ("%d\t%s\n", line, s);
		}
	    }
	  if (r == 1)
	    {
	      free (s);
	      s = InputFileLine (f);
	      line++;
	      free (s);
	      for (int k = 0; k < natom; k++)
		{
		  if ((s = InputFileLine (f)) != NULL)
		    {
		      r = sscanf (s, "%s %lf %lf %lf", atom, &x, &y, &z);
		      addGeometry (j, i, ngeos, k, atom, x, y, z);
		      line++;
		      if (j->debug)
			{
			  printf ("%d\t%s\n", line, s);
			}
		      free (s);
		    }
		}
	      ngeos++;
	    }
	  else
	    {
	      if (j->debug)
		pird ("not read [%s]\n", "geom", s);
	    }
	}
      fclose (f);
    }
  return;
}

void
printGeometries (MULTIJOB * j)
{
  char *tfit[] = LIST_FITS;
  for (int i = 0; i < j->nsys; i++)
    {
      printf ("System %s\n", j->systems[i]);
      for (int k = 0; k < j->data[i]->ne; k++)
	{
	  printf ("\t-Geometry nº %d, energy=%.*g weight=%g\n", k + 1,
		  dg, j->data[i]->energies[k], j->data[i]->weigth[k]);
	  for (int l = 0; l < j->data[i]->n; l++)
	    {
	      printf ("\t %3d %3s %#25.*g %#25.*g %#25.*g\n",
		      l + 1,
		      j->data[i]->geos[k]->atom[l], dg,
		      j->data[i]->geos[k]->x[l], dg,
		      j->data[i]->geos[k]->y[l], dg,
		      j->data[i]->geos[k]->z[l]);
	    }
	}
    }
  printf ("potential:%d\n", j->potential);
  printf ("fitting:%d (%s)\n", j->fitting, tfit[j->fitting]);
}

char *
ReadCoefs (int how)
{
  static char *t1 = "Read and repeat subset.";
  static char *t2 = "Read whole set.";
  if (how)
    return t2;
  return t1;
}

//create a new bounds file to pass to gafit
void
SaveBoundsInter (char *file, vect_domain bounds, int types,
		 int ncoefs, int savemode)
{
  int i, j;
  FILE *out;
  char *mode;
  if (savemode == WRITE)
    mode = "wt";
  else if (savemode == APPEND)
    mode = "at";
  else
    fcStopIt ("Mode unknown");
  if ((out = fopen (file, mode)) != NULL)
    {
      if (savemode == WRITE)
	fprintf (out, "internal communication file\n");
      for (j = 0; j < types; j++)
	{
	  for (i = 0; i < ncoefs; i++)
	    {
	      fprintf (out, "\t%g  %g   %d\n",
		       bounds[j * ncoefs + i].min,
		       bounds[j * ncoefs + i].max,
		       bounds[j * ncoefs + i].decimals);
	    }
	}
      fclose (out);
    }
  else
    {
      printf ("Cannot write file %s\n", file);
      exit (EXIT_FAILURE);
    }
}


void
loadBounds (int rcoefs, int ncoefs, char *external_bounds,
	    int count, vect_domain bounds, int shift)
{
  if (!rcoefs)
    {
      vect_domain tbounds = malloc (sizeof (domain) * ncoefs);
      ReadBounds (tbounds, external_bounds, ncoefs, 1);
      printSepBar (stdout);
      for (int i = 0; i < ncoefs; i++)
	{
	  char *clet = CoefName (i + shift);
	  printf ("\t%10s  %20.17g - %20.17g (%s)\n", clet,
		  tbounds[i].min, tbounds[i].max, ii (tbounds[i].decimals));
	}
      printSepBar (stdout);
      for (int k = 0; k < count; k++)
	{
	  for (int i = 0; i < ncoefs; i++)
	    {
	      bounds[k * ncoefs + i].min = tbounds[i].min;
	      bounds[k * ncoefs + i].max = tbounds[i].max;
	      bounds[k * ncoefs + i].gap = tbounds[i].gap;
	      bounds[k * ncoefs + i].decimals = tbounds[i].decimals;
	    }
	}
      free (tbounds);
    }
  else
    {
      ReadBounds (bounds, external_bounds, ncoefs, count);
      for (int i = 0; i < count * ncoefs; i++)
	{
	  char *clet = CoefName (i % ncoefs + shift);
	  printf ("\t%5d %10s %20.17g - %20.17g (%s)\n", (i + 1),
		  clet, bounds[i].min,
		  bounds[i].max, ii (bounds[i].decimals));
	  if ((i + 1) % ncoefs == 0)
	    printf ("\n");
	}
    }
}


void
mReadBounds (MULTIJOB * j)
{
  char *bfile;
  printf ("\n");
  printf ("BOUNDS\n");
  printSepBar (stdout);
  printf ("\n");
  printf ("Bounds:[%s]\n", j->job->external_bounds);
  printf ("\n");
  printf ("Reading bounds for %d coefficients\n", j->job->ncoefs);
  printf ("\n");
  loadBounds (j->rcoefs, j->job->ncoefs,
	      j->job->external_bounds, j->count2body, j->job->bounds, 0);
  printf ("\n");
  printf (" %d BOUNDS VECTOR:\n", j->job->N);
  printSepBar (stdout);
  printf ("\n");
  PrintBounds (j, j->job->bounds, j->count2body,
	       j->job->ncoefs, 1, j->count2body, '2');
  if (j->tbinteractions)
    {
      printf ("3body Bounds:[%s]\n", j->tbody->external_bounds);
      printf ("\n");
      printf ("Reading bounds for %d coefficients\n", j->tbody->ncoefs);
      printf ("\n");
      loadBounds (j->tbody->rcoefs, j->tbody->ncoefs, j->tbody->external_bounds, j->tbody->count3body, j->tbody->bounds, j->job->ncoefs);	// previous read coefs
      printf ("\n");
      printf ("3BODY %d BOUNDS VECTOR:\n", j->tbody->N);
      printSepBar (stdout);
      printf ("\n");
      PrintBounds (j, j->tbody->bounds, j->tbody->count3body,
		   j->tbody->ncoefs, 1, j->tbody->count3body, '3');
    }

  if (j->hbinteractions)
    {
      int first_coef = j->job->ncoefs;
      if (j->tbinteractions)
	{
	  first_coef += j->tbody->ncoefs;
	}
      printf ("HbBounds:[%s]\n", j->hbonds->external_bounds);
      printf ("\n");
      printf ("Reading bounds for %d coefficients\n", j->hbonds->ncoefs);
      printf ("\n");
      loadBounds (j->hbonds->rcoefs, j->hbonds->ncoefs,
		  j->hbonds->external_bounds, j->hbonds->countHb,
		  j->hbonds->bounds, first_coef);
// printf ("\n");
// printf ("HB %d BOUNDS VECTOR:\n", j->hbonds->N);
// printSepBar (stdout);
// printf ("\n");
// PrintBounds (j, j->hbonds->bounds, j->hbonds->countHb,
//                 j->hbonds->ncoefs, 1, j->hbonds->countHb, 'H');
    }


  bfile = MakePath (j->job->external_bounds, ".internal");
  SaveBoundsInter (bfile, j->job->bounds, j->count2body,
		   j->job->ncoefs, WRITE);
  if (j->tbinteractions)
    {
      SaveBoundsInter (bfile, j->tbody->bounds,
		       j->tbody->count3body, j->tbody->ncoefs, APPEND);
    }
  if (j->hbinteractions)
    {
      SaveBoundsInter (bfile, j->hbonds->bounds,
		       j->hbonds->countHb, j->hbonds->ncoefs, APPEND);
    }
  free (bfile);
}

void
printMultiBanner (MULTIJOB * j)
{
  printf ("\n");
  printf ("MULTI INTERMOLECULAR MODULE\n");
  printSepBar (stdout);
  printPots ();
  printf
    ("Potential read: %s. This potential has %d coefficients per interaction.\n",
     j->potential_name, j->job->ncoefs);
  printf ("Read all coefficients: %s, %s\n",
	  j->rcoefs ? "yes" : "no", ReadCoefs (j->rcoefs));
  printf ("Fitting: ");
  switch (j->fitting)
    {
    case 0:
      printf (" %s\n", AFIT);
      break;
    case 1:
      printf (" %s\n", RFIT);
      break;
    case 2:
      printf (" %s\n", UFIT);
      break;
    default:
      printf (" unknown\n");
      fcStopIt ("Unknown fitting method\n");
    }
  printf ("\n");
}

int
overStrike (char *tape, int pos, char *letters)
{
  int l = strlen (letters);
  for (int i = 0; i < l; i++)
    {
      tape[i + pos] = letters[i];
    }
  return pos + l;
}

void
printResume (MULTIJOB * j)
{

  printf ("\n\n");
  char *bar = "+-------+---------------+-----------+----------------+\n";
  char *bar2 = "+----------------------------------------------------+\n";
  printf (bar2);
  char tyre1[] = "|          TOTAL                                     |\n";
  char tyre2[] = "|                                   |  %10d    |\n";
  int pos1 = 17;
  int pos2 = 2;
  pos1 = overStrike (tyre1, pos1, "2body ");
  pos2 = overStrike (tyre2, pos2, "2 body ");
  if (j->tbinteractions)
    {
      pos1 = overStrike (tyre1, pos1, "+ 3body ");
      pos2 = overStrike (tyre2, pos2, "+ 3 body ");
    }
// if (j->hbinteractions)
//   {
//     pos1 = overStrike (tyre1, pos1, "+ Hb ");
//     pos2 = overStrike (tyre2, pos2, "+ Hb ");
//   }
  overStrike (tyre1, pos1, "INTERACTIONS");
  overStrike (tyre2, pos2, "interactions");
  printf (tyre1);
  printf (bar);
  printf ("|       |  interactions |   coefs   |       total    |\n");
  printf (bar);
  printf ("| 2body |  %10d   | %4d      |  %10d    |\n",
	  j->count2body, j->job->ncoefs, j->job->N);
  if (j->tbinteractions)
    printf ("| 3body |  %10d   | %4d      |  %10d    |\n",
	    j->tbody->count3body, j->tbody->ncoefs, j->tbody->N);
//if (j->hbinteractions)
//  printf ("|   hb  |           1   | %4d      |  %10d    |\n",
//        j->hbonds->ncoefs, j->hbonds->N);
  printf (bar);
//printf (tyre2, j->job->N + j->tbody->N + j->hbonds->N);
  printf (tyre2, j->job->N + j->tbody->N);
  printf (bar);
  printf ("\n");
}

void
printAll (MULTIJOB * j)
{
  printMultiBanner (j);
  printTypes (j);
  printInteractions (j);
  if (j->tbinteractions)
    {
      print3Body (j);
      printResume (j);
    }
  if (j->hbinteractions)
    printHBounds (j);
//  printEnergies (j);
//  printGeometries (j);
  if (j->phbinteractions)
    printPHBounds (j);
}

int
nGeoms (MULTIJOB * j)
{
  int ng = 0;
  for (int i = 0; i < j->nsys; i++)
    ng += j->data[i]->ne;
  return ng;
}

double
functionCaller (int system, int geo, int atoma, int atomb, int debug)
{
  FPOINTER f;			//function pointer to call C
  int fn;			//function number to call Fortran 
  int nr = 1;			// one distance
  double r[nr];
  int na = 2;			// two atoms
  int atoms[na];
  double *vc;			// coefs vector
  int nc;
  char l;			// potential language
  double result;
  if (debug)
    printf ("++function call system: %d, geo: %d, atoma: %d, atomb: %d\n",
	    system, geo, atoma, atomb);
  vc = interVec (system, atoma, atomb);	// load potential coefficients
  nc = nCoeffs ('2');		// number of coefficientes per potential
  r[0] = distance (system, geo, atoma, atomb);	// distannce between atoma atomb
  l = getPotLang ('2');
  atoms[0] = atoma;		// atoms vector
  atoms[1] = atomb;
  switch (l)
    {
    case 'f':
      fn = getPotFunctionF ('2');
      result =
	FortranFunctionRouter (&fn, &system, &geo, &na, atoms, &nr, r, &nc,
			       vc);
      break;
    case 'c':
      f = getPotFunctionC ('2');
      result = f (system, geo, na, atoms, nr, r, nc, vc);
      break;
    default:
      fcStopIt ("Lang potential unknown\n");
    }
  fflush (stdout);
  if (debug)
    {
      char *na, *nb, *ta, *tb;
      int nta, ntb;
      na = atomName (system, atoma);
      ta = atomType (system, atoma);
      nb = atomName (system, atomb);
      tb = atomType (system, atomb);
      nta = atomNType (system, atoma);
      ntb = atomNType (system, atomb);
      printf
	("\tsys:%d geo:%d  atoma:%d %s (type:%s,ntype:%d) atomb:%d %s (type:%s,ntype:%d)",
	 system, geo, atoma, na, ta, nta, atomb, nb, tb, ntb);
      printf (" interac:%s (number:%d)\n",
	      strInter (system, atoma, atomb), numInter (nta, ntb));
      printf ("\tcoefs: ");
      for (int i = 0; i < nCoeffs ('2'); i++)
	{
	  printf ("c%d=%lf ", i, vc[i]);
	}
      printf ("\n");
      printf ("\tcoor: ");
      printf ("x=%lf y=%lf z=%lf / x=%lf y=%lf z=%lf\n",
	      coorX (system, geo, atoma), coorY (system, geo,
						 atoma),
	      coorZ (system, geo, atoma), coorX (system, geo,
						 atomb),
	      coorY (system, geo, atomb), coorZ (system, geo, atomb));
      printf ("\tdistance a-b=%lf\n", r[0]);
      printf ("\tlang=%c name=%s ", l, potName ('2'));
      printf ("\tresult=%lf\n", result);
    }

  free (vc);
  return result;
}

double
function3Caller (int system, int geo, int atoma, int atomb,
		 int atomc, int debug)
{
  FPOINTER f;			// C pointer
  int fn;			// Fortran function number
  int na = 3;			// number of atoms, 3body=3
  int atoms[na];
  int nr = 3;			// distances, 3body=3
  double r[nr];
  double result;
  double *vc;
  int nc;
  char l;
  if (debug)
    printf
      ("++function call system: %d, geo: %d, atoma: %d, atomb: %d, atomc: %d\n",
       system, geo, atoma, atomb, atomc);
  vc = inter3Vec (system, atoma, atomb, atomc);
  nc = nCoeffs ('3');
  r[0] = distance (system, geo, atoma, atomb);
  r[1] = distance (system, geo, atomb, atomc);
  r[2] = distance (system, geo, atomc, atoma);
  l = getPotLang ('3');
  switch (l)
    {
    case 'f':
      fn = getPotFunctionF ('3');
      result =
	FortranFunctionRouter (&fn, &system, &geo, &na, atoms, &nr, r, &nc,
			       vc);
      break;
    case 'c':
      f = getPotFunctionC ('3');
      result = f (system, geo, na, atoms, nr, r, nc, vc);
      break;
    default:
      fcStopIt ("Lang potential unknown\n");
    }
  fflush (stdout);
  if (debug)
    {
      char *na, *nb, *nc, *ta, *tb, *tc;
      int nta, ntb, ntc;
      na = atomName (system, atoma);
      ta = atomType (system, atoma);
      nta = atomNType (system, atoma);
      nb = atomName (system, atomb);
      tb = atomType (system, atomb);
      ntb = atomNType (system, atomb);
      nc = atomName (system, atomc);
      tc = atomType (system, atomc);
      ntc = atomNType (system, atomc);
      printf
	("\tsys:%d geo:%d  atoma:%d %s (type:%s,ntype:%d) atomb:%d %s (type:%s,ntype:%d) atomc:%d %s (type:%s,ntype:%d)",
	 system, geo, atoma, na, ta, nta, atomb, nb, tb, ntb, atomc,
	 nc, tc, ntc);
      printf (" interac:%s (number:%d)\n",
	      str3Inter (system, atoma, atomb, atomc),
	      num3Inter (nta, ntb, ntc));
      printf ("\tcoefs: ");
      for (int i = 0; i < nCoeffs ('3'); i++)
	{
	  printf ("c%d=%lf ", i, vc[i]);
	}
      printf ("\n");
      printf ("\tcoor: ");
      printf
	("x=%lf y=%lf z=%lf / x=%lf y=%lf z=%lf / x=%lf y=%lf z=%lf\n",
	 coorX (system, geo, atoma), coorY (system, geo, atoma),
	 coorZ (system, geo, atoma), coorX (system, geo, atomb),
	 coorY (system, geo, atomb), coorZ (system, geo, atomb),
	 coorX (system, geo, atomc), coorY (system, geo, atomc),
	 coorZ (system, geo, atomc));
      printf ("\tdistance a-b=%lf b-c=%lf c-a=%lf\n", r[0], r[1], r[2]);
      printf ("\tlang=%c name=%s ", l, potName ('3'));
      printf ("\tresult=%lf\n", result);
    }

  free (vc);
  return result;
}

double
functionHbCaller (MULTIJOB * j, int system, int geo, int debug)
{
// https://www.paulnorvig.com/guides/interoperability-of-fortran-with-cc.html
// facer un hbwrapper.f90
  int natoms, ncoefs, frag;	//n atoms, n coefs, last first fragment atom
  char *labels;			// system's atom names (to lower)
  int *ilabels;			// system's atomic numbers
  double *geoxyz;		// geometry
  double *vcoefs;		// hb coefs from 1 to 4 max.
  double *atomcoefs;		// hb coef asigned to atom, 0 if no N or O
  double result = 0;		// hb contribution
  int *map_coefs;		// maps each atom with hb coef
  int kind = j->hbonds->kind;	// kind of hbonds to consider

  natoms = j->hbonds->hbsys[system]->n_atoms;
  frag = j->hbonds->hbsys[system]->frag;
  labels = j->hbonds->hbsys[system]->labels_block;
  ilabels = j->hbonds->hbsys[system]->ilabels_block;

  geoxyz = j->hbonds->hbsys[system]->geo[geo];

  map_coefs = j->hbonds->hbsys[system]->hb_coef;
  ncoefs = j->hbonds->ncoefs;
  vcoefs = interHbVec ();
  atomcoefs = malloc (sizeof (double) * natoms);
  for (int i = 0; i < natoms; i++)
    {
      if (map_coefs[i] == 0)
	{
	  atomcoefs[i] = 0;
	}
      else
	{
	  atomcoefs[i] = vcoefs[map_coefs[i] - 1];
	}
    }

  result =
    HbWrapper (&natoms, labels, ilabels, atomcoefs, geoxyz, &frag, &kind,
	       &debug);

// kcal <- hartree
  result *= HARTREE2KCAL;

  if (j->debug)
    {
      printf ("++function call Hb sys:%d (%s), geo:%d\n", system,
	      j->systems[system], geo);
      printf ("\tncoefs:%d\n", ncoefs);
      for (int i = 0; i < ncoefs; i++)	// coefs from 1 to 4 as passed to fortran, 
	{
	  printf ("\t\t %d %s  %lf\n", i + 1,
		  getHbCoefString (i + 1, j->hbonds), vcoefs[i]);
	}
      printf
	("\tatom \tname \tn_atm \tcoef \tcoef value\t    x\t      y\t        z\n");
      printf
	("\t+--------------------------------------------------------------------------+\n");
      for (int i = 0; i < natoms; i++)
	{
	  printf
	    ("\t%d \t%c%c \t%d \t%d %s\t%11.6lf \t%9.6lf %9.6lf %9.6lf\n",
	     i + 1, labels[i * 2], labels[i * 2 + 1], ilabels[i],
	     map_coefs[i], getHbCoefString (map_coefs[i], j->hbonds),
	     atomcoefs[i], geoxyz[i * 3], geoxyz[i * 3 + 1],
	     geoxyz[i * 3 + 2]);
	}
      printf ("\tatom coordinates in angstrom, energy in kcal/mol\n");
      printf ("\tfirst fragment: %d atoms\n", frag);
      printf ("\tcontribution: %lf\n", result);
    }

  return result;
}

double
EvalGeo (MULTIJOB * j, int system, int geo)
{
  // calculate fit
  double calc = 0;

  j->contrib_2body = 0;
  j->contrib_3body = 0;
  j->contrib_hbonds = 0;

  // 2 body interactions
  //for each atom in frag A
  for (int a = 0; a < j->data[system]->frag; a++)
    {
      //for each atom in frag B
      for (int b = j->data[system]->frag; b < j->data[system]->n; b++)
	{
	  if (j->debug)
	    printf ("\n");
	  j->contrib_2body += functionCaller (system, geo, a, b, j->debug);
	}
    }
  if (j->tbinteractions)
    {
      //
      // 3body interactions here.
      // for each atom a in frag A
      for (int a = 0; a < j->data[system]->frag; a++)
	{
	  // for each atom b in frag B
	  for (int b = j->data[system]->frag; b < j->data[system]->n; b++)
	    {
	      // for each atom c in frag B c != b
	      for (int c = b + 1; c < j->data[system]->n; c++)
		{
		  if (j->debug)
		    printf ("\n");
		  j->contrib_3body +=
		    function3Caller (system, geo, a, b, c, j->debug);
		}
	    }
	}
      // reverse, from frag B to A
      // for each atom a in frag B
      for (int a = j->data[system]->frag; a < j->data[system]->n; a++)
	{
	  // for each atom b in frag A
	  for (int b = 0; b < j->data[system]->frag; b++)
	    {
	      // for each atom c in frag A c != b
	      for (int c = b + 1; c < j->data[system]->frag; c++)
		{
		  if (j->debug)
		    printf ("\n");
		  j->contrib_3body +=
		    function3Caller (system, geo, a, b, c, j->debug);
		}
	    }
	}
    }

  if (j->hbinteractions)
    {
      if (j->debug)
	printf ("\n");
      j->contrib_hbonds = functionHbCaller (j, system, geo, j->hbonds->debug);
    }

  calc = j->contrib_2body + j->contrib_3body + j->contrib_hbonds;

  return calc;
}

double
fitGeo (MULTIJOB * j, int system, int geo)
{
  double vvalue = j->data[system]->energies[geo];
  double cvalue = EvalGeo (j, system, geo);
  double fitness;
  double delta2 = (vvalue - cvalue) * (vvalue - cvalue);
  if (j->fitting == 0)
    {
      fitness = delta2 * j->data[system]->weigth[geo];
    }
  else if (j->fitting == 1)
    {
      fitness =
	delta2 / (j->data[system]->energies[geo] *
		  j->data[system]->energies[geo]) *
	j->data[system]->weigth[geo];
    }
  return fitness;
}

double
eval_multi (MULTIJOB * j, double *genes)
{
  double fitness = 0;
  //double delta = 0;
  setMPointer (j);
  setCoeffs (genes);
  if (j->debug)
    {
      printf ("----eval multi--------\n");
    }
  fitness = 0;
  // for each system
  for (int i = 0; i < j->nsys; i++)
    {
      // for each system's geometry 
      for (int k = 0; k < j->data[i]->ne; k++)
	{
	  if (j->debug)
	    printf ("+system:%4d geometry:%4d\n", i, k);
	  // calculate one geometry
	  fitness += fitGeo (j, i, k);
	  if (j->debug)
	    {
	      printf ("+fitting=%lg\n", fitness);
	      printf ("+system:%d geometry:%d accumulated:%lf\n", i, k,
		      fitness);
	    }
	}
    }
  if (j->debug)
    printf ("==== fitting=%lg\n", fitness);
  return fitness;
}

/*
 * to change dynamicaly debug mode
 */
void
setDebug (MULTIJOB * j, int how)
{
  j->debug = how;
}

char *
getString (char *f, char *sec, char *it)
{
  char temp[STRING_MAX];
  if (!GetStringParameter (f, sec, it, "", temp, STRING_MAX))
    {
      fcPrintfStopIt ("Error %s %s\n", sec, it);
    }
  return strdup (temp);
}

// print Input Reading Debug
void
pird (char *format, ...)
{
  va_list args;
  char prompt[] = "read debug: %20s:\t";
  int n = strlen (prompt);
  int m = strlen (format);
  char buffer[n + m + 1];
  va_start (args, format);
  memcpy (buffer, prompt, n + 1);
  memcpy (buffer + n, format, m + 1);
  vprintf (buffer, args);
  va_end (args);
  fflush (stdout);
}

// debug mark pass
void
dmp (char *mark)
{
  pird ("%s\n", mark, "-------pass-------");
}

void
MultiAnalyzer (MULTIJOB * j)
{
  char temp[STRING_MAX];
  char **tokens = nllBuild (",", NULL);
  char *geo, *ener, *a2t;
  char *tfit[] = LIST_FITS;
  char *hbkind[] = LIST_HB_KINDS;
  char *hbalgo[] = LIST_HB_ALGO;
  int ncoefs2plusx = 0;

  // set debug
  j->debug = GetBoolParameter (j->job->file, "multi", "debug", DEFAULT_DEBUG);
  // 3body interactions
  j->tbinteractions =
    GetBoolParameter (j->job->file, "multi", "3body",
		      DEFAULT_3B_INTERACTIONS);

  int xbonds = GetFromSetParameter (j->job->file, "multi", "hbonds", hbalgo,
				    HB_ALGO_DEFAULT, 0);

  j->hbinteractions = j->phbinteractions = 0;

  switch (xbonds)
    {
    case HB_ALGO_NONE:
      break;
    case HB_ALGO_HB:
      j->hbinteractions = 1;
      break;
    case HB_ALGO_PHB:
      j->phbinteractions = 1;
      break;
    default:
      fcStopIt ("Error: hb algo set\n");;
    }

  // print machine readable summary
  j->msummary =
    GetBoolParameter (j->job->file, "multi", "msummary", DEFAULT_SUMMARY);
  GetStringParameter (j->job->file, "multi", "systems", "", temp, STRING_MAX);
  j->systems = nllDelete (nllListParser (temp, tokens), tokens);
  j->nsys = nllCount (j->systems);

  if (j->debug)
    pird ("%d\n", "#systems", j->nsys);

  dimData (j);

  for (int i = 0; i < j->nsys; i++)
    {

      sSLtrim (j->systems[i]);
      sSRtrim (j->systems[i]);
      geo = getString (j->job->file, nllString (j->systems, i), "geometries");
      j->geometries = nllAddDup (j->geometries, geo);
      ener = getString (j->job->file, nllString (j->systems, i), "energies");
      j->energies = nllAddDup (j->energies, ener);
      a2t = getString (j->job->file, nllString (j->systems, i), "atom2type");
      j->atom2type = nllAddDup (j->atom2type, a2t);
      if (j->debug)
	pird ("%s / %s / %s\n", j->systems[i], geo, ener, a2t);
    }

  j->potential_name = getString (j->job->file, "multi", "potential");
  setPotential ('2', j->potential_name);
  if (j->debug)
    pird ("%s\n", "potential", j->potential_name);
  j->potential = getPotNumber ('2');
  j->job->ncoefs = getPotCoefs ('2');
  ncoefs2plusx = j->job->ncoefs;
  if (j->debug)
    pird ("%d\n", "#ncoefs", j->job->ncoefs);

  j->rcoefs =
    GetBoolParameter (j->job->file, "job", "all coefficients",
		      ALL_COEFFICIENTS);
  j->fitting =
    GetFromSetParameter (j->job->file, "multi", "fitting", tfit, 1, 0);

  if (j->debug)
    dmp ("job.txt");

  readAtom2Type (j);
  if (j->debug)
    dmp ("atom2types");

  if (j->tbinteractions)
    {
      j->tbody = malloc (sizeof (TBODY));
      j->tbody->potential_name =
	getString (j->job->file, "3body", "potential");
      setPotential ('3', j->tbody->potential_name);
      j->tbody->potential = getPotNumber ('3');
      j->tbody->ncoefs = getPotCoefs ('3');
      ncoefs2plusx += j->tbody->ncoefs;
      j->tbody->external_bounds = getString (j->job->file, "3body", "bounds");
      j->tbody->rcoefs =
	GetBoolParameter (j->job->file, "3body", "all coefficients",
			  ALL_COEFFICIENTS);

      analyze3body (j);

      if (j->debug)
	{
	  dmp ("3body");
	}
    }

  if (j->hbinteractions)
    {
      j->hbonds = malloc (sizeof (HBONDS));
      j->hbonds->external_bounds =
	getString (j->job->file, "hbonds", "bounds");
      j->hbonds->rcoefs =
	GetBoolParameter (j->job->file, "hbonds", "all coefficients",
			  ALL_COEFFICIENTS);
      j->hbonds->debug =
	GetBoolParameter (j->job->file, "hbonds", "debug", DEFAULT_HB_DEBUG);
      j->hbonds->kind =
	GetFromSetParameter (j->job->file, "hbonds", "kind", hbkind,
			     HB_KIND_DEFAULT, 0);
      j->hbonds->ncoefs = 0;

      analyzeHb (j);

    }

  if (j->phbinteractions)
    {
      j->phbonds = malloc (sizeof (PHBONDS));
      j->phbonds->external_bounds =
	getString (j->job->file, "phbonds", "bounds");
      j->phbonds->rcoefs =
	GetBoolParameter (j->job->file, "phbonds", "all coefficients",
			  ALL_COEFFICIENTS);
      j->phbonds->debug =
	GetBoolParameter (j->job->file, "phbonds", "debug", DEFAULT_HB_DEBUG);
      j->phbonds->kind =
	GetFromSetParameter (j->job->file, "phbonds", "kind", hbkind,
			     HB_KIND_DEFAULT, 0);

      analyzePhb (j);

    }

  // add 2body
  j->job->N = j->job->ncoefs * j->count2body;
  j->job->bounds = calloc (1, sizeof (domain) * j->job->N);

  // add 3body
  if (j->tbinteractions != 0)
    {
      j->tbody->N = j->tbody->ncoefs * j->tbody->count3body;
      j->tbody->bounds = calloc (1, sizeof (domain) * j->tbody->N);
    }

  // add hb 
  if (j->hbinteractions != 0)
    {
      j->hbonds->N = j->hbonds->ncoefs * j->hbonds->countHb;
      j->hbonds->bounds = calloc (1, sizeof (domain) * j->hbonds->N);
    }

//    ..OR..

  // add phb 
  if (j->phbinteractions != 0)
    {
      j->phbonds->N = j->phbonds->countPhb;
      j->phbonds->bounds = calloc (1, sizeof (domain) * j->phbonds->N);
    }


  setCNames (j, ncoefs2plusx);


  readEnergies (j);
  if (j->debug)
    dmp ("energies");

  readGeometries (j);

  if (j->debug)
    dmp ("geometries");

  printAll (j);

  if (j->debug)
    dmp ("printAll");

  saveData (j, DATA_FILE);

  if (j->debug)
    dmp ("saveData");

  fcStopIt ("phb not implemented\n");

}

void
setCNames (MULTIJOB * j, int ncoefs2plusx)
{
// setup coefficient's names.///////////////////////////
  // 2body+3body names
  cnFile (j->job->file, ncoefs2plusx);

  //printCnames ();

  if (j->hbinteractions)
    cnAdd (getHbCoefStringS (j->hbonds));
  if (j->phbinteractions)
    cnAdd (getPhbCoefStringS (j->phbonds));

  //printCnames ();
}
