/*
(c) GAFit toolkit $Id: hbonds.c 562 2025-12-11 15:34:12Z ro $
*/

#define _GNU_SOURCE
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "../job.h"
#include "../parameters/parameters.h"
#include "../nullist/nllist.h"
#include "../flyctl/flyctl.h"
#include "../rstrings/rstrings.h"
#include "multijob.h"
#include "multi.h"
#include "hbonds.h"

char **hbcoefnames = NULL;
char **phbcoefnames = NULL;

char *element[] = { "h", "he",
  "li", "be", "b", "c", "n", "o", "f", "ne",
  "na", "mg", "al", "si", "p", "s", "cl", "ar",
  "k", "ca", "sc", "ti", "v", "cr", "mn", "fe", "co", "ni", "cu",
  "zn", "ga", "ge", "as", "se", "br", "kr",
  "rb", "sr", "y", "zr", "nb", "mo", "tc", "ru", "rh", "pd", "ag",
  "cd", "in", "sn", "sb", "te", "i", "xe",
  "cs", "ba", "la", "ce", "pr", "nd", "pm", "sm", "eu", "gd", "tb", "dy",
  "ho", "er", "tm", "yb", "lu", "hf", "ta", "w", "re", "os", "ir", "pt",
  "au", "hg", "tl", "pb", "bi", "po", "at", "rn",
  "fr", "ra", "ac", "th", "pa", "u", "np", "pu"
};



void
hbBanner (void)
{
  printf ("%s", HB_BANNER);
}

char **
GetList (char *file, char *section, char *what)
{
  char **tokens = nllBuild (",", " ", NULL);
  char temp[STRING_MAX];
  char **result = NULL;

  GetStringParameter (file, section, what, "", temp, STRING_MAX);
  if (temp[0])
    {
      result = nllListParser (temp, tokens);
      result = nllDelete (result, tokens);
      return result;
    }
  return NULL;
}


int
setHbCoef (char *type, HBONDS * hbonds)
{
  for (int i = 0; i < hbonds->ncoefs; i++)
    {
      for (int k = 0; k < hbonds->hb_groups[i]->n_types; k++)
	{
	  if (strcmp (hbonds->hb_groups[i]->a_types[k], type) == 0)
	    {
	      return i + 1;	//beginning in 1 to pass to fortran
	    }
	}
    }
  return 0;			// not O or N, no coefficient
}

char *
getHbCoefString (int n, HBONDS * hbonds)
{
  if (n <= 0)
    return "-";
  return hbonds->hb_groups[n - 1]->hbg_name;
}

char **
getHbCoefStringS (HBONDS * hbonds)
{
  if (hbcoefnames != NULL)
    {
      return hbcoefnames;
    }
  for (int i = 0; i < hbonds->ncoefs; i++)
    hbcoefnames = nllAddDup (hbcoefnames, hbonds->hb_groups[i]->hbg_name);
  return hbcoefnames;
}


void
analyzeHb (MULTIJOB * j)
{
  char **hbgroups = NULL;
  hbgroups = GetList (j->job->file, "hbonds", "coefficients groups");
  j->hbonds->ncoefs = nllCount (hbgroups);
  j->hbonds->countHb = 1;	// only one type of interaction HB
  if (j->hbonds->ncoefs == 0)
    fcStopIt ("No coefficients groups!\n");
  j->hbonds->hb_groups = malloc (j->hbonds->ncoefs * sizeof (HBGROUP *));

  for (int i = 0; i < j->hbonds->ncoefs; i++)
    {
      j->hbonds->hb_groups[i] = malloc (sizeof (HBGROUP));
      j->hbonds->hb_groups[i]->hbg_name = nllStringDup (hbgroups, i);
      j->hbonds->hb_groups[i]->a_types =
	GetList (j->job->file, "hbonds", j->hbonds->hb_groups[i]->hbg_name);
      j->hbonds->hb_groups[i]->n_types =
	nllCount (j->hbonds->hb_groups[i]->a_types);
    }

  j->hbonds->nsys = j->nsys;
  j->hbonds->hbsys = malloc (sizeof (HBSYSTEMS *) * j->hbonds->nsys);

  for (int i = 0; i < j->nsys; i++)
    {
      // dimension: number of atoms in each system

      j->hbonds->hbsys[i] = malloc (sizeof (HBSYSTEMS));

      j->hbonds->hbsys[i]->n_atoms = j->data[i]->n;
      j->hbonds->hbsys[i]->hb_coef =
	malloc (sizeof (int) * j->hbonds->hbsys[i]->n_atoms);
      j->hbonds->hbsys[i]->type_n =
	malloc (sizeof (int) * j->hbonds->hbsys[i]->n_atoms);
      j->hbonds->hbsys[i]->atoms =
	malloc (sizeof (char *) * j->hbonds->hbsys[i]->n_atoms);
      j->hbonds->hbsys[i]->types =
	malloc (sizeof (char *) * j->hbonds->hbsys[i]->n_atoms);


      // assign hb coefficient number to each atom or 0 
      for (int k = 0; k < j->hbonds->hbsys[i]->n_atoms; k++)
	{
	  j->hbonds->hbsys[i]->hb_coef[k] = 0;	//default
	  j->hbonds->hbsys[i]->atoms[k] = strdup (j->data[i]->a2tatom[k]);
	  j->hbonds->hbsys[i]->types[k] = strdup (j->data[i]->a2ttype[k]);
	  j->hbonds->hbsys[i]->type_n[k] = j->data[i]->a2tntype[k];
	  j->hbonds->hbsys[i]->hb_coef[k] =
	    setHbCoef (j->hbonds->hbsys[i]->types[k], j->hbonds);
	}
    }
}

void
printHBounds (MULTIJOB * j)
{
  char *hbkinds[] = LIST_HB_KINDS;

  printf ("\n\n");


  printf ("Hydrogen Bonds\n");
  printSepBar (stdout);
  hbBanner ();
  printSepBar (stdout);

  printf ("\n");
  printf ("There are %d coefficients:\n", j->hbonds->ncoefs);
  printf ("\n");
  printf (" coefs.     atom types\n");
  printf ("+-------+-----------------------\n");

  for (int i = 0; i < j->hbonds->ncoefs; i++)
    {
      printf (" %s:    ", j->hbonds->hb_groups[i]->hbg_name);
      for (int k = 0; k < j->hbonds->hb_groups[i]->n_types; k++)
	{
	  printf (" %s ", nllString (j->hbonds->hb_groups[i]->a_types, k));
	}
      printf ("\n");
    }
  printf ("+-------+-----------------------\n");
  printf ("Hb kind:%s\n", hbkinds[j->hbonds->kind]);

  printf ("\n");
  printf ("HB details\n");
  printSepBar (stdout);

  for (int i = 0; i < j->nsys; i++)
    {
      printf ("\n");
      printf ("System: %s\n", j->systems[i]);
      printf ("+-----+------+----------+------+---------+\n");
      printf ("|   n | atom |   type   | ntype| hb coef |\n");
      printf ("+-----+------+----------+------+---------+\n");
      for (int k = 0; k < j->data[i]->n; k++)
	{
	  char *tmp = fws (8, j->hbonds->hbsys[i]->types[k]);
	  char *tmp2 = fws (7,
			    getHbCoefString (j->hbonds->hbsys[i]->hb_coef[k],
					     j->hbonds));
	  printf ("|%4d |  %2s  | %s | %4d | %s |\n", k + 1,
		  j->hbonds->hbsys[i]->atoms[k], tmp,
		  j->hbonds->hbsys[i]->type_n[k] + 1, tmp2);
	  free (tmp);
	  free (tmp2);
	}
      printf ("+-----+------+----------+------+---------+\n");
    }
}

//assign atomic number to atom
int
atomicNumber (char *atom)
{
  for (int i = 0; i < 94; i++)
    {
      if (strcasecmp (atom, element[i]) == 0)
	{
	  return i + 1;
	}
    }
  return 0;
}

// it gives the atom group number from a type 
int
ntype2ngroup (PHBONDS * phb, int type)
{
  return phb->dtypes2agroup[type];
}

// it gives the atom group's name from a type
char *
ntype2strgroup (PHBONDS * phb, int type)
{
  static char *nogroup = " -- ";
  return
    phb->dtypes2agroup[type] ==
    0 ? nogroup : phb->phb_groups[phb->dtypes2agroup[type] - 1]->phbg_name;
}

// it gives the atom group number from system atom number
int
natom2ngroup (PHBONDS * phb, int sys, int n_atom)
{
  return ntype2ngroup (phb, phb->phbsys[sys]->type_n[n_atom]);
}

// it gives the atom group's name from system atom number
char *
natom2strgroup (PHBONDS * phb, int sys, int n_atom)
{
  return ntype2strgroup (phb, phb->phbsys[sys]->type_n[n_atom]);
}

void
buildPairMatrix (PHBONDS * phb, int sys, int atoma, int atomb)
{

  int atype = phb->phbsys[sys]->type_n[atoma];
  int btype = phb->phbsys[sys]->type_n[atomb];
  int agroup = ntype2ngroup (phb, atype);
  int bgroup = ntype2ngroup (phb, btype);
  char *agname = ntype2strgroup (phb, atype);
  char *bgname = ntype2strgroup (phb, btype);
  if (agroup != 0 && bgroup != 0)
    {

      if (phb->debug)
	printf
	  ("sys: %d atoma:%d n:%d(%s) a group: %d(%s) atomb:%d n:%d(%s) b group: %d(%s)\n",
	   sys,
	   atoma, atype, phb->phbsys[sys]->types[atoma], agroup, agname,
	   atomb, btype, phb->phbsys[sys]->types[atomb], bgroup, bgname);

      phb->pairsMat[agroup - 1][bgroup - 1] = 1;
      phb->pairsMat[bgroup - 1][agroup - 1] = 1;
    }
}

int
countAndEnumeratePairs (PHBONDS * phb)
{
  int coef_count = 0;
  for (int i = 0; i < phb->n_groups; i++)
    {
      for (int k = i; k < phb->n_groups; k++)
	{
	  if (phb->pairsMat[i][k] != 0)
	    {
	      coef_count++;
	      phb->pairsMat[i][k] = coef_count;
	      phb->pairsMat[k][i] = coef_count;
	    }
	}
    }
  return coef_count;
}

char *
getPhbCoefString (int n, PHBONDS * phbonds)
{
  if (n <= 0)
    return "-";
  return phbonds->phbStrings[n - 1];
}

char **
getPhbCoefStringS (PHBONDS * phbonds)
{
  if (phbcoefnames != NULL)
    {
      return phbcoefnames;
    }
  for (int i = 0; i < phbonds->ncoefs; i++)
    phbcoefnames = nllAddDup (phbcoefnames, phbonds->phbStrings[i]);
  return phbcoefnames;
}

void
analyzePhb (MULTIJOB * j)
{
  static char labels[STRING_MAX];
  char **phbgroups = NULL;
  phbgroups = GetList (j->job->file, "phbonds", "atom groups");
  j->phbonds->n_groups = nllCount (phbgroups);

  if (j->phbonds->n_groups == 0)
    fcStopIt ("No atoms groups!\n");

  j->phbonds->phb_groups = malloc (j->phbonds->n_groups * sizeof (HBGROUP *));

  for (int i = 0; i < j->phbonds->n_groups; i++)
    {
      j->phbonds->phb_groups[i] = malloc (sizeof (PHBGROUP));
      j->phbonds->phb_groups[i]->phbg_name = nllStringDup (phbgroups, i);
      j->phbonds->phb_groups[i]->a_types =
	GetList (j->job->file, "phbonds",
		 j->phbonds->phb_groups[i]->phbg_name);
      j->phbonds->phb_groups[i]->n_types =
	nllCount (j->phbonds->phb_groups[i]->a_types);
    }

  j->phbonds->dtypes2agroup = malloc (sizeof (int) * j->dtypes);

  if (j->phbonds->debug)
    {
      printf ("Atom groups of interest\n");
      printf ("--------------------------\n");
    }
  for (int l = 0; l < j->dtypes; l++)
    {
      j->phbonds->dtypes2agroup[l] = 0;
      if (j->phbonds->debug)

	printf ("%d - %s\n", l + 1, j->types[l]);
      for (int i = 0; i < j->phbonds->n_groups; i++)
	{
	  if (j->phbonds->debug)
	    printf ("\t%d-%s: ", i + 1, j->phbonds->phb_groups[i]->phbg_name);
	  for (int k = 0; k < j->phbonds->phb_groups[i]->n_types; k++)
	    {
	      if (j->phbonds->debug)
		printf (" %s", j->phbonds->phb_groups[i]->a_types[k]);
	      if (strcmp
		  (j->types[l], j->phbonds->phb_groups[i]->a_types[k]) == 0)
		{
		  if (j->phbonds->debug)
		    printf ("\n\t\t\t%s - %s match!:%d\n", j->types[i],
			    j->phbonds->phb_groups[i]->phbg_name, i + 1);
		  j->phbonds->dtypes2agroup[l] = i + 1;
		  break;
		}
	    }
	  if (j->phbonds->debug)
	    printf ("\n");
	  if (j->phbonds->dtypes2agroup[l] != 0)
	    break;
	}
    }
  if (j->phbonds->debug)
    printf ("\n");



  printf ("Map between atoms and phb groups\n");
  printf ("+----------+---------+---------+------------+\n");
  printf ("|  number  |  atom   | n group | group name |\n");
  printf ("+----------+---------+---------+------------+\n");
  for (int i = 0; i < j->dtypes; i++)
    {
      char *gname =
	j->phbonds->dtypes2agroup[i] ==
	0 ? "--" : j->phbonds->phb_groups[j->phbonds->dtypes2agroup[i] -
					  1]->phbg_name;
      printf ("| %5d    | %7s | %5d   |   %7s  |\n", i + 1, j->types[i],
	      j->phbonds->dtypes2agroup[i], gname);
    }
  printf ("+----------+---------+---------+------------+\n");

  j->phbonds->pairsMat = malloc (sizeof (int *) * j->phbonds->n_groups);
  for (int i = 0; i < j->phbonds->n_groups; i++)
    j->phbonds->pairsMat[i] = malloc (sizeof (int) * j->phbonds->n_groups);

  for (int i = 0; i < j->phbonds->n_groups; i++)
    for (int k = 0; k < j->phbonds->n_groups; k++)
      j->phbonds->pairsMat[i][k] = 0;	// no pair


  j->phbonds->nsys = j->nsys;
  j->phbonds->phbsys = malloc (sizeof (PHBSYSTEMS *) * j->phbonds->nsys);

  j->phbonds->countPhb = 0;

  for (int i = 0; i < j->nsys; i++)
    {

      j->phbonds->phbsys[i] = malloc (sizeof (PHBSYSTEMS));

      j->phbonds->phbsys[i]->n_atoms = j->data[i]->n;
      j->phbonds->phbsys[i]->frag = j->data[i]->frag;

      /*   j->phbonds->phbsys[i]->phb_coef =
         malloc (sizeof (int) * j->phbonds->phbsys[i]->n_atoms); */
      j->phbonds->phbsys[i]->type_n =
	malloc (sizeof (int) * j->phbonds->phbsys[i]->n_atoms);
      j->phbonds->phbsys[i]->atoms =
	malloc (sizeof (char *) * j->phbonds->phbsys[i]->n_atoms);
      j->phbonds->phbsys[i]->types =
	malloc (sizeof (char *) * j->phbonds->phbsys[i]->n_atoms);
      j->phbonds->phbsys[i]->pg_name =
	malloc (sizeof (char *) * j->phbonds->phbsys[i]->n_atoms);
      j->phbonds->phbsys[i]->pg_n =
	malloc (sizeof (int) * j->phbonds->phbsys[i]->n_atoms);

      // study of atom pairs to map to each pair's coefficient

      // assign hb coefficient number to each atom or 0 
      for (int k = 0; k < j->phbonds->phbsys[i]->n_atoms; k++)
	{
	  /*  j->hbonds->hbsys[i]->hb_coef[k] = 0;        //default */
	  j->phbonds->phbsys[i]->atoms[k] = strdup (j->data[i]->a2tatom[k]);
	  j->phbonds->phbsys[i]->types[k] = strdup (j->data[i]->a2ttype[k]);
	  j->phbonds->phbsys[i]->type_n[k] = j->data[i]->a2tntype[k];
	  //j->phbonds->phbsys[i]->pg_n[k];
	  //j->phbonds->phbsys[i]->pg_name[k]; 

	  if (j->phbonds->debug)
	    printf ("k:%d atom:%s type:%s ntype:%d\n", k,
		    j->data[i]->a2tatom[k], j->data[i]->a2ttype[k],
		    j->data[i]->a2tntype[k]);
	  /*  j->hbonds->hbsys[i]->hb_coef[k] =
	     setHbCoef (j->hbonds->hbsys[i]->types[k], j->hbonds);
	   */
	}


      switch (j->phbonds->kind)
	{
	case HB_KIND_INTRA:
	  if (j->phbonds->debug)
	    printf ("Checking intra pairs");
	  fcStopIt ("phbonds kind not implemented");
	  break;
	case HB_KIND_INTER:
	  if (j->phbonds->debug)
	    printf ("Checking inter pairs\n");
	  for (int k = 0; k < j->phbonds->phbsys[i]->frag; k++)
	    for (int l = j->phbonds->phbsys[i]->frag;
		 l < j->phbonds->phbsys[i]->n_atoms; l++)
	      {
		buildPairMatrix (j->phbonds, i, k, l);
	      }
	  break;
	case HB_KIND_ALL:
	  if (j->phbonds->debug)
	    printf ("Checking all pairs");
	  for (int k = 0; k < j->phbonds->phbsys[i]->n_atoms; k++)
	    for (int l = k + 1; l < j->phbonds->phbsys[i]->n_atoms; l++)
	      {
		buildPairMatrix (j->phbonds, i, k, l);
	      }

	  break;
	default:
	  fcStopIt ("phbonds kind invalid!\n");
	  printf ("\n");
	}
    }

  j->phbonds->countPhb = j->phbonds->ncoefs = j->phbonds->N =
    countAndEnumeratePairs (j->phbonds);
  j->phbonds->phbStrings =
    (char **) malloc (sizeof (char *) * j->phbonds->countPhb);
  printf ("\n");
  printf ("There are %d atom-pairs so there are %d coefficients\n\n",
	  j->phbonds->countPhb, j->phbonds->countPhb);

  for (int i = 0; i < j->phbonds->n_groups; i++)
    {
      for (int k = i; k < j->phbonds->n_groups; k++)
	{
	  int coef = j->phbonds->pairsMat[i][k] - 1;
	  snprintf (labels, STRING_MAX, "%s-%s",
		    j->phbonds->phb_groups[i]->phbg_name,
		    j->phbonds->phb_groups[k]->phbg_name);
	  j->phbonds->phbStrings[coef] = strdup (labels);
	}
    }
  printf ("COEFFICIENT PAIRS' SUMMARY MATRIX\n");
  printf
    ("+----------+------------------------+------------------------+------------------------+------------------------+\n");
  printf ("|          ");
  for (int i = 0; i < j->phbonds->n_groups; i++)
    {
      printf ("|       %10s       ", j->phbonds->phb_groups[i]->phbg_name);
    }
  printf ("|\n");
  printf
    ("+----------+------------------------+------------------------+------------------------+------------------------+\n");

  for (int i = 0; i < j->phbonds->n_groups; i++)
    {
      printf ("| %8s ", j->phbonds->phb_groups[i]->phbg_name);
      for (int k = 0; k < j->phbonds->n_groups; k++)
	{
	  if (k < i)
	    {
	      printf ("|%*s", 8 + 14 + 2, "");
	    }
	  else
	    {
	      int coef = j->phbonds->pairsMat[i][k];
	      printf ("| %8d %-14s", coef,
		      getPhbCoefString (coef, j->phbonds));
	    }
	}
      printf ("|\n");
    }

  printf
    ("+----------+------------------------+------------------------+------------------------+------------------------+\n");

}

void
printPHBounds (MULTIJOB * j)
{
  char *hbkinds[] = LIST_HB_KINDS;

  printf ("\n\n");


  printf ("Hydrogen Bonds\n");
  printSepBar (stdout);
  hbBanner ();
  printSepBar (stdout);

  printf ("\n");
  printf ("There are %d coefficients:\n", j->hbonds->ncoefs);
  printf ("\n");
  printf (" coefs.     atom types\n");
  printf ("+-------+-----------------------\n");

  for (int i = 0; i < j->hbonds->ncoefs; i++)
    {
      printf (" %s:    ", j->hbonds->hb_groups[i]->hbg_name);
      for (int k = 0; k < j->hbonds->hb_groups[i]->n_types; k++)
	{
	  printf (" %s ", nllString (j->hbonds->hb_groups[i]->a_types, k));
	}
      printf ("\n");
    }
  printf ("+-------+-----------------------\n");
  printf ("Hb kind:%s\n", hbkinds[j->hbonds->kind]);

  printf ("\n");
  printf ("HB details\n");
  printSepBar (stdout);

  for (int i = 0; i < j->nsys; i++)
    {
      printf ("\n");
      printf ("System: %s\n", j->systems[i]);
      printf ("+-----+------+----------+------+---------+\n");
      printf ("|   n | atom |   type   | ntype| hb coef |\n");
      printf ("+-----+------+----------+------+---------+\n");
      for (int k = 0; k < j->data[i]->n; k++)
	{
	  char *tmp = fws (8, j->hbonds->hbsys[i]->types[k]);
	  char *tmp2 = fws (7,
			    getHbCoefString (j->hbonds->hbsys[i]->hb_coef[k],
					     j->hbonds));
	  printf ("|%4d |  %2s  | %s | %4d | %s |\n", k + 1,
		  j->hbonds->hbsys[i]->atoms[k], tmp,
		  j->hbonds->hbsys[i]->type_n[k] + 1, tmp2);
	  free (tmp);
	  free (tmp2);
	}
      printf ("+-----+------+----------+------+---------+\n");
    }
}
