/*
 (c)GAFit toolkit $Id: final.c 561 2025-12-10 15:24:48Z ro $
*/
#if HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <float.h>
#include "../core.h"
#include "../inputline/line.h"
#include "../rand.h"
#include "../literals/literals.h"
#include "../rstrings/rstrings.h"
#include "../cnames.h"
#include "../job.h"
#include "multijob.h"
#include "auxfuncs.h"
#include "../cnames.h"
#include "final.h"

void
PrintOne (FILE * outfile, double *coeffs, int ncoefs, int j, int i,
	  int coefshift, int indexshift)
{
  int index = j * ncoefs + i + indexshift;
  fprintf (outfile, "\t%5d %10s %+15.5lf\n", index + 1,
	   CoefName (i + coefshift), coeffs[index]);

}

void
PrintCoefficients (FILE * outfile, MULTIJOB * jo, double *genes, char kind)
{
  int count, ncoefs, last, coefshift, indexshift;
  count = ncoefs = last = coefshift = indexshift = 0;

  switch (kind)
    {
    case '2':
      count = jo->count2body;
      ncoefs = jo->job->ncoefs;
      last = jo->job->ncoefs;
      coefshift = 0;
      indexshift = 0;
      break;
    case '3':
      count = jo->tbody->count3body;
      ncoefs = jo->tbody->ncoefs;
      last = jo->tbody->ncoefs;
      coefshift = jo->job->ncoefs;
      indexshift = jo->count2body * jo->job->ncoefs;
      break;
    case 'H':
      count = jo->hbonds->countHb;
      ncoefs = jo->hbonds->ncoefs;
      last = jo->hbonds->ncoefs;
      coefshift = jo->job->ncoefs;
      if (jo->tbinteractions)
	coefshift += jo->tbody->ncoefs;
      indexshift = jo->count2body * jo->job->ncoefs;
      if (jo->tbinteractions)
	indexshift += jo->tbody->count3body * jo->tbody->ncoefs;
      break;
    default:
      return;
    }

  for (int j = 0; j < count; j++)
    {
      fprintf (outfile, "INTERACTION TYPE %d ( %s )\n", j + 1,
	       strInterTypeAll (j + 1, kind));
      fprintf (outfile, "-----------------------------");
      fprintf (outfile, "\n\tCoefficients:\n");
      for (int i = 0; i < last; i++)
	{
	  PrintOne (outfile, genes, ncoefs, j, i, coefshift, indexshift);
	}
    }
}

unsigned int
readSeed ()
{
  unsigned int seed;
  FILE *f = fopen (RANDOM_SAVE_FILE, "r");
  fscanf (f, "%u", &seed);
  fclose (f);
  return seed;
}

void
scissors (FILE * out)
{
  fprintf (out,
	   "-----8<----------------------------------------------------------------------------------------------------\n");
}

void
MachineSummary (FILE * outfile, MULTIJOB * j, double *genes, double fit)
{
  unsigned int seed = readSeed ();
  fprintf (outfile, "\n");
  scissors (outfile);
  fprintf (outfile, ":seed: %u :fit: %lf\n", seed, fit);
  for (int i = 0; i < j->count2body; i++)
    {
      for (int k = 0; k < j->job->ncoefs; k++)
	{
	  int index = i * j->job->ncoefs + k;
	  fprintf (outfile,
		   ":n: %5d :int: %5d :type: %-15s  :coeff: %5d  :name: %-10s :value: %+.*lf\n",
		   index + 1, i + 1, strInterTypeAll (i + 1, '2'),
		   k + 1, CoefName (k), DBL_DECIMAL_DIG, genes[index]);
	}
    }
  if (j->tbinteractions)
    {
      for (int i = 0; i < j->tbody->count3body; i++)
	{
	  for (int k = 0; k < j->tbody->ncoefs; k++)
	    {
	      int index =
		i * j->tbody->ncoefs + k + j->count2body * j->job->ncoefs;
	      fprintf (outfile,
		       ":n: %5d :int: %5d :type: %-15s  :coeff: %5d  :name: %-10s :value: %+.*lf\n",
		       index + 1, i + 1, strInterTypeAll (i + 1, '3'), k + 1,
		       CoefName (k + j->job->ncoefs), DBL_DECIMAL_DIG,
		       genes[index + k]);
	    }
	}
    }
  if (j->hbinteractions)
    {
      for (int i = 0; i < j->hbonds->countHb; i++)
	{
	  for (int k = 0; k < j->hbonds->ncoefs; k++)
	    {
	      int index =
		i * j->hbonds->ncoefs + k + j->count2body * j->job->ncoefs;
	      if (j->tbinteractions)
		{
		  index += j->tbody->count3body * j->tbody->ncoefs;
		}
	      fprintf (outfile,
		       ":n: %5d :int: %5d :type: %-15s  :coeff: %5d  :name: %-10s :value: %+.*lf\n",
		       index + 1, i + 1, strInterTypeAll (i + 1, 'H'), k + 1,
		       CoefName (k + j->job->ncoefs +
				 (j->tbinteractions ? j->tbody->ncoefs : 0)),
		       DBL_DECIMAL_DIG, genes[index]);

	    }
	}
    }

  scissors (outfile);
}

void
FinalEvaluation (FILE * outfile, MULTIJOB * j)
{
  double fit;
  double *genes = LoadBest (OUTPUT_BEST, NC (j));

  setMPointer (j);

  fprintf (outfile, "\n\n#\n#Results\n#\n");
  PrintCoefficients (outfile, j, genes, '2');

  if (j->tbinteractions)
    PrintCoefficients (outfile, j, genes, '3');

  if (j->hbinteractions)
    PrintCoefficients (outfile, j, genes, 'H');

  fit = Evaluation (outfile, j, genes);

  if (j->msummary)
    MachineSummary (outfile, j, genes, fit);
}

double
Evaluation (FILE * outfile, MULTIJOB * j, double *genes)
{
  double cvalue, vvalue, weigth;
  int indexs, indexg;
  double fitabsolute;

  fprintf (outfile, "#\n#\nSystems:\n");
  for (int i = 0; i < j->nsys; i++)
    printf ("%5d - %s\n", i + 1, j->systems[i]);
  fprintf (outfile, "#\n");
  fprintf (outfile, "#\n#Evaluation\n");
  fprintf (outfile, "#\n");
  fprintf (outfile,
	   "#System  Geometry  Reference       Calculated      Difference      Weight       2Body         ");
  if (j->tbinteractions)
    fprintf (outfile, "  3body         ");
  if (j->hbinteractions)
    fprintf (outfile, "  hbonds        ");
  printf ("\n");
  fprintf (outfile,
	   "#======  ========  ==============  ==============  ==============  ===========  ==============");

  if (j->tbinteractions)
    fprintf (outfile, "  ==============");
  if (j->hbinteractions)
    fprintf (outfile, "  ==============");
  printf ("\n");

  setMPointer (j);
  //sets the coefficients for calculations
  setCoeffs (genes);

  fitabsolute = 0;

  for (int i = 0; i < j->nsys; i++)	//systems
    {
      indexs = i + 1;
      for (int k = 0; k < j->data[i]->ne; k++)	// geometries
	{
	  indexg = k + 1;
	  cvalue = EvalGeo (j, i, k);
	  vvalue = j->data[i]->energies[k];
	  weigth = j->data[i]->weigth[k];

	  fprintf (outfile,
		   " %5d     %5d  %+14.3lf  %+14.3lf   %+12.2lf %%      %+6.2lf  %+14.3lf",
		   indexs, indexg, vvalue, cvalue,
		   (cvalue - vvalue) / vvalue * 100, weigth,
		   j->contrib_2body);

	  if (j->tbinteractions)
	    fprintf (outfile, "  %+14.3lf ", j->contrib_3body);

	  if (j->hbinteractions)
	    fprintf (outfile, " %+14.3lf ", j->contrib_hbonds);

	  printf ("\n");

	  fitabsolute += fitGeo (j, i, k);
	}
    }
  printf ("\n");
  printf ("Fit absolute: %lf\n", fitabsolute);
  return fitabsolute;
}

double *
LoadBest (char *filename, int howmany)
{
  FILE *best;
  double *x;
  x = (double *) malloc (sizeof (double) * howmany);
  if ((best = fopen (OUTPUT_BEST, "rt")) != NULL)
    {
      for (int i = 0; i < howmany; i++)
	{
	  fscanf (best, "%lf\n", &x[i]);
	  // printf ("%d %lf\n",i, x[i]);
	}
      fclose (best);
    }
  return x;
}
