/*
   (c) GAFit toolkit $Id: cpotentials.c 505 2025-01-25 02:06:51Z ro $
*/

#if HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <math.h>
#include "../job.h"
#include "multijob.h"
#include "auxfuncs.h"
#include "regpots.h"


double
cexp1 (int system, int geo, int na, int *atoms, int nr, double *r,
       int nc, double *vc)
{
  return vc[0] * exp (-vc[1] * r[0]) + vc[2] / pow (r[0], vc[3]);
}

double
cexp2 (int system, int geo, int na, int *atoms, int nr, double *r,
       int nc, double *vc)
{
  return vc[0] * exp (-vc[1] * r[0]) +
    vc[2] / pow (r[0], vc[3]) + vc[4] / pow (r[0], vc[5]);
}

double
cexp3 (int system, int geo, int na, int *atoms, int nr, double *r,
       int nc, double *vc)
{
  return vc[0] * exp (-vc[1] * r[0]) +
    vc[2] / pow (r[0], vc[3]) +
    vc[4] / pow (r[0], vc[5]) + vc[6] / pow (r[0], vc[7]);
}

////////////  ATM  3body potential Axilrod-Teller-Muto  //////////// 

/// reading rshort from job.txt and caching its value
double
RShort ()
{
  static float rsvalue = 0.0;
  if (rsvalue == 0.0)
    rsvalue = getParameter ("3body", "rshort", 1.5);
  //default: 1.5
  return rsvalue;
}

/// decay upon rshort
double
decayAtm (double r)
{
  double rshort = RShort ();
  if (r > rshort)
    return 1.0;
  return exp (-100.0 * pow (rshort / r - 1.0, 2.0));
}

/// ATM pot
double
catm (int system, int geo, int na, int *atoms, int nr, double *r,
      int nc, double *vc)
{
  // na=3 atoms , nr=3 distances, nc=1 coefficient
  // distances between atom j-k, k-m, m-j:
  double rjk = r[0];
  double rkm = r[1];
  double rmj = r[2];
  double c9 = vc[0];
  double rjk2 = rjk * rjk;
  double rkm2 = rkm * rkm;
  double rmj2 = rmj * rmj;
  double cosjk = (rkm2 + rmj2 - rjk2) / (2 * rkm * rmj);
  double coskm = (rjk2 + rmj2 - rkm2) / (2 * rjk * rmj);
  double cosmj = (rjk2 + rkm2 - rmj2) / (2 * rjk * rkm);

  double v = 3 * c9 * (1 + 3 * cosjk * coskm * cosmj)
    / (rjk2 * rjk * rkm2 * rkm * rmj2 * rmj);

  return v * decayAtm (rjk) * decayAtm (rkm) * decayAtm (rmj);
}

//////////// end ATM ///////////


// this must be the last, otherwise declare each potential
// at the beginning:
//
//  double cexpX(int system,int geo, int na, int *atoms, int nr, double *r, int nc, double *vc);
//  ..
//
//  double cexpX(...){
//  ...code...
//  }
//  ...
// 
//  and the location of CSetupPots can be any within the file
//  after the declarations.
//
void
CSetupPots (void)
{
  CRegisterPot ("cexp1", cexp1, 4, '2');
  CRegisterPot ("cexp2", cexp2, 6, '2');
  CRegisterPot ("cexp3", cexp3, 8, '2');
  CRegisterPot ("catm", catm, 1, '3');
}
