/*
   (c) GAFit toolkit $Id: auxfuncs.c 561 2025-12-10 15:24:48Z ro $
*/

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>

#include "../job.h"
#include "multijob.h"
#include "../parameters/parameters.h"
#include "3body.h"
#include "hbonds.h"
#include "auxfuncs.h"

/*
 * all the funtions in this file are based on this 
 * pointer.
 * set with #setMPointer(MULTIJOB *j)
 */
MULTIJOB *jp = NULL;

/*
 * sets the current multijob.
 * @param the multijob pointer
 */
void
setMPointer (MULTIJOB * j)
{
  jp = j;
}

/*
 * sets the coefficients vector
 * @param coeffs the coefficients.
 */
void
setCoeffs (double *coeffs)
{
  jp->vec_coeffs = coeffs;
}

/*
 * @return the atom type string
 * @param system system number
 * @atom atom atom number
 */
char *
atomType (int system, int atom)
{
  return jp->data[system]->a2ttype[atom];
}

/*
 * @return the atom type number
 * @param system number
 * @param atom atom number
 */
int
atomNType (int system, int atom)
{
  return jp->data[system]->a2tntype[atom];
}

/*
 * @return atom name string
 * @param system system number
 * @param atom atom number
 */
char *
atomName (int system, int atom)
{
  return jp->data[system]->a2tatom[atom];
}

double
coorX (int system, int geo, int atom)
{
  return jp->data[system]->geos[geo]->x[atom];
}

double
coorY (int system, int geo, int atom)
{
  return jp->data[system]->geos[geo]->y[atom];
}

double
coorZ (int system, int geo, int atom)
{
  return jp->data[system]->geos[geo]->z[atom];
}

double
distance (int system, int geo, int atoma, int atomb)
{
  double r, dx, dy, dz;
  dx = coorX (system, geo, atoma) - coorX (system, geo, atomb);
  dy = coorY (system, geo, atoma) - coorY (system, geo, atomb);
  dz = coorZ (system, geo, atoma) - coorZ (system, geo, atomb);
  r = sqrt (dx * dx + dy * dy + dz * dz);
  return r;
}

/* 
 * Returns the interaction string for 2body the interaction 
 * on the very first stage.
 *
 * @param interaction the interaction number, 
 * starting at 1.
 * @return the interaction string
 *
 **/

char *
strInterType (int interaction)
{
  static char response[STRING_MAX];
  for (int i = 0; i < jp->dtypes; i++)
    {
      for (int j = 0; j < jp->dtypes; j++)
	{

	  int index = mat2vec (jp, i, j);
	  int interac = jp->interactions[index];
	  if (interac == interaction)
	    {
	      sprintf (response, "%s-%s", jp->types[i], jp->types[j]);
	      return response;
	    }
	}
      sprintf (response, "NO INTERACTION");
    }
  return response;
}


void
PrintAnyInteractions (char kind)
{
  int final = kind == '2' ? jp->count2body : jp->tbody->count3body;

  for (int i = 0; i < final; i++)
    {
      switch (kind)
	{
	case '2':
	  printf ("interaction %d typea %d typeb %d string %s\n", i,
		  jp->dbodyint[i].typea, jp->dbodyint[i].typeb,
		  jp->dbodyint[i].str);
	  break;
	case '3':
	  printf ("interaction %d typea %d typeb %d typec %d string %s\n", i,
		  jp->tbody->tbodyint[i].typea, jp->tbody->tbodyint[i].typeb,
		  jp->tbody->tbodyint[i].typec, jp->tbody->tbodyint[i].str);
	  break;
	}
    }
}


/*
 * returns the interaction string after created with
 * strInterType, saved and load again.
 * @param interaction interaction number;
 * @return interaction string.
 */
char *
strInterTypeAll (int interaction, char kind)
{
  char *ret;

  switch (kind)
    {
    case '2':
      ret = jp->dbodyint[interaction - 1].str;
      break;
    case '3':
      ret = jp->tbody->tbodyint[interaction - 1].str;
      break;
    case 'H':
      ret = HB_INTERACTION;
      break;
    default:
      ret = "NO INTERACTION";
      break;
    }
  return ret;
}

/* 
 * Returns the atom types for one interaction.
 *
 * @param interaction the interaction number, 
 * starting at 1.
 *
 * @return the atom types string and the 
 * atom types in parameters atomtype1 and
 * atomtype2
 *
 **/
char *
strInterAtomType (int interaction, int *atomtype1, int *atomtype2)
{
  static char response[STRING_MAX];
  for (int i = 0; i < jp->dtypes; i++)
    {
      for (int j = 0; j < jp->dtypes; j++)
	{

	  int index = mat2vec (jp, i, j);
	  int interac = jp->interactions[index];
	  //printf("i=%d j=%d index=%d, inter=%d \n",i,j,index,interac);
	  if (interac == interaction)
	    {
	      sprintf (response, "%4d  %4d", i + 1, j + 1);
	      *atomtype1 = i + 1;
	      *atomtype2 = j + 1;
	      return response;
	    }
	}
      sprintf (response, "NO INTERACTION");
    }
  return response;
}

/*
 * Returns the interaction string for 
 * the interaction between atom a and b from
 * any system.
 */
char *
strInter (int system, int atoma, int atomb)
{
  static char res[STRING_MAX];
  res[0] = '\0';
  strcat (res, atomType (system, atoma));
  strcat (res, "-");
  strcat (res, atomType (system, atomb));
  return res;
}

char *
str3Inter (int system, int atoma, int atomb, int atomc)
{
  static char res[STRING_MAX];
  res[0] = '\0';
  strcat (res, atomType (system, atoma));
  strcat (res, "-");
  strcat (res, atomType (system, atomb));
  strcat (res, "-");
  strcat (res, atomType (system, atomc));
  return res;
}

/*
 * returns interaction number (from 0 to n-1)
 * @param typea number of atom type
 * @param typeb number of atom type
 * @return number of interaction (from 0 to n-1)
 */
int
numInter (int typea, int typeb)
{
  return mat2vec (jp, typea, typeb);
}

int
num3Inter (int typea, int typeb, int typec)
{
  int a, b, c;
  a = typea;
  b = typeb;
  c = typec;
  cubeOrder (&a, &b, &c);
  return jp->tbody->type2intCube[a][b][c];
}

/*
 * Returns the interaction's coefficients vector
 * for the interaction between atom a and atom b 
 * from some system.
 *
 * @return interaction's coefficients vector
 * @param system[in] system system number
 * @param atoma[in] first atom
 * @param atomb[in] second atom
 */
double *
interVec (int system, int atoma, int atomb)
{
  int num_interaction =
    numInter (atomNType (system, atoma), atomNType (system, atomb));

  int coefs_row = jp->interactions[num_interaction] - 1;	//coefs row in matrix coefs

  double *vcoefs = (double *) malloc (sizeof (double) * jp->job->ncoefs);
  if (jp->debug)
    {
      int j = 0;
      printf ("\tinteraction %d, row %d coefficient's matrix: ",
	      num_interaction, coefs_row);
      for (int i = 0; i < jp->job->N; i++)
	{
	  if ((j % jp->job->ncoefs) == 0)
	    printf ("\n\t[interaction %d]  ", j / jp->job->ncoefs);
	  j++;
	  printf (" %lf", jp->vec_coeffs[i]);
	}
      printf ("\n");
    }

  for (int i = 0; i < jp->job->ncoefs; i++)
    {
      vcoefs[i] = jp->vec_coeffs[coefs_row * jp->job->ncoefs + i];
    }
  return vcoefs;
}

double *
inter3Vec (int system, int atoma, int atomb, int atomc)
{
//----------------------------------------//
  int num_3interaction =
    num3Inter (atomNType (system, atoma), atomNType (system, atomb),
	       atomNType (system, atomc));
  int first3bodycoeff = jp->job->N + num_3interaction * jp->tbody->ncoefs;

  double *vcoefs = (double *) malloc (sizeof (double) * jp->tbody->ncoefs);

  //printf("system:%d atoma:%d (%d) atomb:%d (%d) atomc:%d (%d)  n3int:%d first:%d\n",system,atoma,atomNType(system,atoma), atomb,atomNType(system,atomb),atomc,atomNType(system,atomc),num_3interaction, first3bodycoeff);

  if (jp->debug)
    {
      int j = 0;
      printf
	("\t3body interaction: %d, 2body coefficients: %d, so 3body coefficients begin in: %d line",
	 num_3interaction, jp->job->N, first3bodycoeff);
      for (int i = 0; i < jp->tbody->N; i++)
	{
	  if ((j % jp->tbody->ncoefs) == 0)
	    printf ("\n\t[interaction %d]  ", j / jp->tbody->ncoefs);
	  j++;
	  printf (" %lf", jp->vec_coeffs[i]);
	}
      printf ("\n");
    }
  for (int i = 0; i < jp->tbody->ncoefs; i++)
    {
      vcoefs[i] = jp->vec_coeffs[first3bodycoeff + i];
    }
  return vcoefs;
}

/**
 * there is only one Hb interaction 
 * with a maximum of  4 distinct coefs
 */
double *
interHbVec ()
{
  int firstHbCoef = jp->job->N;
  if (jp->tbinteractions)
    firstHbCoef += jp->tbody->N;

  double *vcoefs = (double *) malloc (sizeof (double) * jp->hbonds->N);
  for (int i = 0; i < jp->hbonds->ncoefs; i++)
    {
      vcoefs[i] = jp->vec_coeffs[firstHbCoef + i];
    }
  return vcoefs;
}

/*
 * @parameter kind interactions type, '2' or '3'
 * @return number of coefficients per interaction
 */
int
nCoeffs (char kind)
{
  if (kind == '2')
    return jp->job->ncoefs;
  if (kind == '3')
    return jp->tbody->ncoefs;
  return -1;
}

/*
 * @parameter kind interactions type, '2' or '3'
 * @returns the potential name in use
 */
char *
potName (char kind)
{
  if (kind == '2')
    return jp->potential_name;
  if (kind == '3')
    return jp->tbody->potential_name;
  return NULL;
}

double
getParameter (char *section, char *parameter, double value)
{
  return GetDblParameter (jp->job->file, section, parameter, value);
}
