/*
(c) GAFit toolkit $Id: shepherd.c 510 2025-04-22 14:23:13Z ro $
*/
#if HAVE_CONFIG_H
#include <config.h>
#endif
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <errno.h>
#if defined(_POSIX_TIMERS) && _POSIX_TIMERS > 0
#include <time.h>
#else
#include <sys/time.h>
#endif
#include "../inputline/line.h"
#include "../flyctl/flyctl.h"
#include "../literals/literals.h"
#include "../rstrings/rstrings.h"
#include "../nullist/nllist.h"
#include "mopac.h"


extern char *MopacLicense;
extern char *MopacExecutable;
extern char *MopacMop;
extern char *MopacTemplate;
extern int ToolsOutput;
extern int NCores;

int
shClock_gettime (struct timespec *ts)
{
#if defined(_POSIX_TIMERS) && _POSIX_TIMERS > 0
//  return clock_gettime (CLOCK_REALTIME, &ts);
  return clock_gettime (CLOCK_REALTIME, ts);
#else
  struct timeval tv;
  gettimeofday (&tv, NULL);
  ts->tv_sec = tv.tv_sec;
  ts->tv_nsec = tv.tv_usec * 1000;
  return 0;
#endif
}

char *
moExtFile (char *file, char *ext)
{
  char *s = (char *) malloc (sizeof (char) * strlen (file) + 1);
  strcpy (s, file);
  sReplaceString (s, ".mop", ext);
  return s;
}

char **
moList (char *file)
{
  char **list = NULL;
  char *l[] = MOPAC_F_LIST;
  char **p;
  p = l;
  while (*p)
    {
      list = nllAdd (list, moExtFile (file, *p));
      p++;
    }
  return list;
}

int
moUnLink (char **files)
{
  char *s;
  while ((s = nllPop (&files)) != NULL)
    unlink (s);
  nllClear (files);
  return 1;
}

int
moDelete (char *file)
{
  char **s;
  s = moList (file);
  s = nllAdd (s, moExtFile (file, ".out"));
  s = nllAdd (s, strdup (file));
  moUnLink (s);
  return 1;
}

int
LastSheep (char *file)
{
  FILE *f;
  char *s, *p, *q;
  int lastSheep = 0;
  f = fopen (file, "r");
  while ((s = InputFileLine (f)) != NULL)
    {
      if ((p = strstr (s, "Sheep #")))
	{
	  p = strstr (s, "#") + 1;
	  q = strstr (p, "#");
	  q[0] = 0;
	  lastSheep = Srettel (p);
	}
      free (s);
    }
  fclose (f);
  return lastSheep;
}

int
FirstSheep (char *file)
{
  FILE *f;
  char *s, *p, *q;
  int firstSheep = 0;
  f = fopen (file, "r");
  while ((s = InputFileLine (f)) != NULL)
    {
      if ((p = strstr (s, "Sheep #")))
	{
	  p = strstr (s, "#") + 1;
	  q = strstr (p, "#");
	  q[0] = 0;
	  firstSheep = Srettel (p);
	  free (s);
	  fclose (f);
	  return firstSheep;
	}
      free (s);
    }
  fclose (f);
  return -1;
}

char *
SplitFlock (char *flockfile, int first, int last)
{
  char *sfirst = Letters (first);
  char *slast = Letters (last);
  char *sfile =
    (char *) malloc (sizeof (char) *
		     (strlen (sfirst) + strlen (slast) + strlen (".mop") +
		      2));
  strcpy (sfile, sfirst);
  strcat (sfile, "-");
  strcat (sfile, slast);
  strcat (sfile, ".mop");

  CreateMopacMop (first, last + 1, MopacTemplate, sfile);

  free (sfirst);
  free (slast);

  return sfile;
}

void
FirstSheepFail (char *file, int sheep, int ncalc)
{
  FILE *f;
  int i;
  char *mk;
  mk = malloc (sizeof (char) * (strlen (SHEEP_MARK) + 1));
//  printf ("FirstSheepFail: %s\n", file);
  //write the marks recognized by extractor
  f = fopen (file, "w");
  for (i = 0; i < ncalc; i++)
    {
      fprintf (f, "%s\n\n", MOPAC_SITE);
      strcpy (mk, SHEEP_MARK);
      mk = sReplaceString (mk, "@", Letters (sheep));
      fprintf (f, "%s\n", mk);
      fprintf (f, "%s\n\n\n", NON_RECOVERABLE_ERROR);
      fprintf (f, "%s\n\n\n", MOPAC_DONE);
    }
  free (mk);
  fclose (f);
}

void
Glue (char *oldfile, char *newfile)
{
  FILE *f;
  char *s;
  char **buffer = NULL;
//  printf ("gluing %s with %s\n", oldfile, newfile);
  f = fopen (newfile, "r");
  while ((s = InputFileLine (f)) != NULL)
    {
      buffer = nllAdd (buffer, strdup (s));
      free (s);
    }
  fclose (f);

  buffer = nllReverse (buffer);

  f = fopen (oldfile, "a");
  while ((s = nllPop (&buffer)) != NULL)
    {
      fprintf (f, "%s\n", s);
    }
  fclose (f);
  nllClear (buffer);

}


void
Consolidate (char *oldfile, char *newfile, int lostsheep)
{
  FILE *f;
  char *s;
  char **buffer = NULL;
  char *mk;
  mk = malloc (sizeof (char) * (strlen (SHEEP_MARK) + 1));
  strcpy (mk, SHEEP_MARK);
  mk = sReplaceString (mk, "@", Letters (lostsheep));
#ifdef __DEBUG__
  printf ("consolidate oldfile:%s newfile:%s lostsheep:%d\n", oldfile,
	  newfile, lostsheep);
#endif
  f = fopen (oldfile, "r");
  if (f)
    {
      while ((s = InputFileLine (f)) != NULL)
	{
	  buffer = nllAdd (buffer, strdup (s));
	  //search for the lost sheep
	  if (strstr (s, mk))
	    {
	      free (s);
	      break;
	    }
	  free (s);
	}
      fclose (f);
      //cut from lost sheep to last sheep
      while (!strstr (nllLook (buffer), MOPAC_DONE))
	nllPopLost (&buffer);

      //add lost sheep's flock
      f = fopen (newfile, "r");
      while ((s = InputFileLine (f)) != NULL)
	{
	  buffer = nllAdd (buffer, strdup (s));
	  free (s);
	}
      fclose (f);

      buffer = nllReverse (buffer);

      f = fopen (oldfile, "w");
      while ((s = nllPop (&buffer)) != NULL)
	{
	  fprintf (f, "%s\n", s);
	}
      fclose (f);
      nllClear (buffer);
    }
}


int
FlockCtl (char *license, char *executable, char *inputfile)
{
  pid_t flockpid;
  int errpipe[2];
  int ostderr;
  char input[2048];
  int sheep, lsheep;
  int r;
  char *arguments[3];

  sheep = FirstSheep (inputfile);
  lsheep = LastSheep (inputfile);

  arguments[2] = NULL;
  arguments[1] = inputfile;
  arguments[0] =
    (char *) malloc (sizeof (char) *
		     (strlen (license) + 1 + strlen (executable) + 1));
  strcpy (arguments[0], license);
  strcat (arguments[0], "/");
  strcat (arguments[0], executable);

  if (!pipe (errpipe))
    {
      ostderr = dup (2);
      close (2);
      dup2 (errpipe[1], 2);
      if (!(flockpid = fork ()))
	{
	  setenv ("LIBC_FATAL_STDERR_", "1", 1);
	  setenv ("MOPAC_LICENSE", license, 1);
	  close (errpipe[0]);
	  close (errpipe[1]);
	  if (execv (arguments[0], arguments))
	    {
	      int a = errno;
	      char er[SNPRINTF];
	      perror (arguments[0]);
	      snprintf (er, SNPRINTF - 1, "%s - %s\n", arguments[0],
			strerror (a));
	      fcStopIt (er);
	    }
	}
      else
	{
	  close (2);

	  dup2 (ostderr, 2);

	  close (errpipe[1]);

	  r = read (errpipe[0], input, 2048);
	  input[r] = 0;
	  if (r > 0)
	    {
	      /// hack for MOPAC 2016 part 1
	      /// on finishing a job, MOPAC 2016 writes to *stderr*
	      /// a message like: "MOPAC Job: "mopac_input.mop" ended normally on Aug  5, 2016, at 13:21."
	      ///  
	      if (strstr (input, "ended normally") == NULL)
		{
		  /// end hack part 1
		  char *ofile, *nwfile;
		  char *newflock;
		  int fsheep;
		  char run_command[SNPRINTF];
#ifdef __DEBUG__
		  printf ("shepherd errno %d %s\n", errno, input);
#endif
		  //let child write __STOP__ file 
		  sleep (1);
		  //check for __STOP__ file
		  fcCheck ();

		  kill (flockpid, SIGKILL);
		  ofile = moExtFile (arguments[1], ".out");
		  fsheep = LastSheep (ofile);
#ifdef __DEBUG__
		  printf ("in file %s lost sheep:%d\n", ofile, fsheep);
#endif
		  if (sheep == fsheep)
		    {
		      int ncalc = CountCalcs (MopacTemplate);
		      FirstSheepFail (ofile, fsheep, ncalc);
		      fsheep++;
		    }
		  if (lsheep > fsheep)
		    {
		      newflock = SplitFlock (inputfile, fsheep, lsheep);
		      snprintf (run_command, SNPRINTF - 1, "%s %s",
				"shepherd", newflock);
		      system (run_command);
		      nwfile = moExtFile (newflock, ".out");
		      Consolidate (ofile, nwfile, fsheep);
		      free (newflock);
		      free (nwfile);
		    }
		  free (ofile);
		}
	      /// hack for MOPAC 2016 part 2
	      /// MOPAC's ended normally processes continue here...
	    }
	  /// end hack part a
	  // barrier if overflow in read pipe
	  waitpid (flockpid, NULL, 0);
	}
      free (arguments[0]);
      return 0;
    }
  else
    {
      fcStopIt ("shepherd: pipe call doesn't work!\n");
    }
  return 1;
}

void
usage (void)
{
  printf ("shepherd v0.2 (c)GAFit toolkit - 2012,2018\n");
  printf ("\tUsage: shepherd [mopac-input-file]\n\n");
}

void
writeFlockTimes (SATIME * flckt, int *lastn)
{
  FILE *ft;
  ft = fopen (FILE_FTIMES, "w");
  fwrite (lastn, sizeof (int), 1, ft);
  fwrite (flckt, sizeof (SATIME), FLOCK_MAX, ft);
  fclose (ft);
}

void
readFlockTimes (SATIME * flckt, int *lastn)
{
  FILE *ft;
  ft = fopen (FILE_FTIMES, "r");
  fread (lastn, sizeof (int), 1, ft);
  fread (flckt, sizeof (SATIME), FLOCK_MAX, ft);
  fclose (ft);
}

void
loadFlockTimes (SATIME * flckt, int *lastn)
{
  int i;

  if (FileExist (FILE_FTIMES))
    readFlockTimes (flckt, lastn);
  else
    {
      *lastn = 0;
      for (i = 0; i < FLOCK_MAX; i++)
	{
	  flckt[i].t = 0;
	  flckt[i].n = 0;
	  flckt[i].max = 0;
	  flckt[i].min = 0;
	}
      writeFlockTimes (flckt, lastn);
    }
}

void
forker (char *license, char *executable, char *newflock)
{
  pid_t pid;
  pid = fork ();
  if (pid == 0)
    {
      FlockCtl (license, executable, newflock);
      exit (EXIT_SUCCESS);
    }
  if (pid == -1)
    {
      printf ("shepherd fork failed\n");
      char er[SNPRINTF];
      snprintf (er, SNPRINTF - 1, "shepherd fork failed\n");
      fcStopIt (er);
    }
}

int
MachineGun (char *license, char *executable, char *inputfile, int burst)
{
  int fsheep, esheep;
  int total;
  char **newflock;
  char *ofile, *nwfile;

  struct timespec start, end;
  double elapsed;
  int timeok = 1;
  SATIME flock_time[FLOCK_MAX];
  int lastn;
  int i, n, pieces, nn;

  FILE *f;

  total = LastSheep (inputfile) + 1;

  if (burst)
    newflock = (char **) malloc (sizeof (char *) * FLOCK_MAX);
  else
    newflock = (char **) malloc (sizeof (char *) * total);

  loadFlockTimes (flock_time, &lastn);

  pieces = FLOCK_MAX;
  n = FLOCK_MAX;

  if (NCores == 0)
    {
      nn = 0;

      for (i = 0; i < FLOCK_MAX; i++)
	nn += flock_time[i].n;

      for (i = 0; i < FLOCK_MAX; i++)
	{
	  if (flock_time[i].t == 0)
	    {
	      n = i + 1;
	      break;
	    }
	  if ((i + 1) != FLOCK_MAX)
	    if (flock_time[i].t < flock_time[i + 1].t)
	      {
		if (nn % 3 == 0)
		  n = i + 1;
		else if (nn % 3 == 1)
		  n = i + 2;
		else if (nn % 3 == 2)
		  {
		    n = i;
		    if (n == 0)
		      n = 1;
		  }
		break;
	      }
	}
    }
  else
    {
      n = NCores;
    }

  if (shClock_gettime (&start) == -1)
    {
      timeok = 0;
      n = 1;
    }

  if (n > total)
    n = total;

  writeFlockTimes (flock_time, &n);

  if (ToolsOutput)
    {
      printf ("shepherd #flocks:%d\n", n);
      printf ("ncores=%d\n", NCores);
    }
  fflush (stdout);
  fflush (stderr);

  // n controls the running total processes in parallel
  for (i = 0; i < n; i++)
    {
      if (burst)
	{
	  fsheep = i * total / n;
	  esheep = (i + 1) * total / n - 1;
	  newflock[i] = SplitFlock (inputfile, fsheep, esheep);
	}
      else
	{
	  newflock[i] = SplitFlock (inputfile, i, i);
	}
      forker (license, executable, newflock[i]);
    }

  while (1)
    {
      int status;
      pid_t done = wait (&status);
      if (done == -1)
	{
	  if (errno == ECHILD)
	    break;		// no more child processes
	}
      else
	{
	  if (!WIFEXITED (status) || WEXITSTATUS (status) != 0)
	    {
	      char er[SNPRINTF];
	      snprintf (er, SNPRINTF - 1, "pid %d failed\n", done);
	      fcStopIt (er);
	    }
	  //one more
	  if (!burst)
	    {
	      if (i < total)
		{
		  newflock[i] = SplitFlock (inputfile, i, i);
		  forker (license, executable, newflock[i]);
		  i++;
		}
	    }
	}
    }

  //clearing file
  nwfile = moExtFile (inputfile, ".out");
  f = fopen (nwfile, "w");
  fclose (f);
  free (nwfile);

  if (!burst)
    pieces = total;

  ofile = moExtFile (inputfile, ".out");
  for (i = 0; i < pieces; i++)
    {
      nwfile = moExtFile (newflock[i], ".out");
      Glue (ofile, nwfile);
      moDelete (newflock[i]);
      free (nwfile);
    }
  free (ofile);

  for (i = 0; i < total; i++)
    {
      nwfile = Letters (i);
      unlink (nwfile);
      free (nwfile);
    }

  if (shClock_gettime (&end) == -1)
    timeok = 0;

  if (timeok)
    {
      elapsed = (end.tv_sec - start.tv_sec)
	+ (end.tv_nsec - start.tv_nsec) / 1e9;

      if (ToolsOutput)
	printf ("shepherd elapsed time:%lf\n", elapsed);

      flock_time[n - 1].n++;
      flock_time[n - 1].t = elapsed;
      if (flock_time[n - 1].min > elapsed || flock_time[n - 1].min == 0)
	flock_time[n - 1].min = elapsed;
      if (flock_time[n - 1].max < elapsed || flock_time[n - 1].max == 0)
	flock_time[n - 1].max = elapsed;
      lastn = n;
      writeFlockTimes (flock_time, &lastn);
      return 1;
    }
  return 0;
}


/*usermanualbegin*/
int
main (int argc, char **argv)
{
  int burst = 0;		//* \label{line:shepherdmain}
/*usermanualend*/
  fcCheck ();
  InitEnvironmentVariables ();
  if (argc > 1)
    {
      if (argc > 2 || strcmp (argv[1], "-h") == 0)
	usage ();
      else
	{
	  FlockCtl (MopacLicense, MopacExecutable, argv[1]);
	}
    }
  else
    {
      MachineGun (MopacLicense, MopacExecutable, MopacMop, burst);
    }
  return 0;
}
