/*
(c) GAFit toolkit $Id: input.c 510 2025-04-22 14:23:13Z ro $
*/
#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include "../core.h"
#include "../job.h"
#include "bounds.h"
#include "../inputline/line.h"
#include "../parameters/parameters.h"
#include "../nullist/nllist.h"
#include "../analytical/analytical.h"
#include "../cnames.h"
#include "../numbers.h"
#include "../rstrings/rstrings.h"
#include "interface.h"
#include "intpot.h"

char *
ReadCoefs (int how)
{
  static char *t1 = "Read and repeat subset";
  static char *t2 = "Read whole set";
  if (how)
    return t2;
  return t1;
}

int
ReadFIJob (JOB * jo, char *f1, char *f2, char *f3, char *f4, char *fb,
	   int *pot, int *rcoef, int *prtgeo, int *fitting,
	   int *prtan, int *inttypes, int *nint)
{
  char *tfit[] = LIST_FITS;
  char *tinter[] = LIST_INTR_TYPE;
  if (FileExist (jo->file))
    {
      GetStringParameter (jo->file, "job", "geometries", "geometries.txt", f1,
			  STRING_MAX);
      GetStringParameter (jo->file, "job", "energies", "energies.txt", f2,
			  STRING_MAX);
      GetStringParameter (jo->file, "job", "atom2type", "atom2type.txt", f3,
			  STRING_MAX);
      GetStringParameter (jo->file, "job", "bounds", "bounds.txt", fb,
			  STRING_MAX);
      GetStringParameter (jo->file, "job", "charges", "charges.txt", f4,
			  STRING_MAX);
      *pot = GetIntParameter (jo->file, "job", "potential", 1);

      //jorge
      *nint = GetIntParameter (jo->file, "job", "interactions", 0);
      if (*nint)
	*inttypes = nllLocate ("user-defined", tinter);
      else
	*inttypes =
	  GetFromSetParameter (jo->file, "job", "interactions", tinter,
			       INTR_TYPE, 0);
      //jorge

      *rcoef =
	GetBoolParameter (jo->file, "job", "all coefficients",
			  ALL_COEFFICIENTS);
      *prtgeo =
	GetBoolParameter (jo->file, "print", "geometries", PRINT_GEOMETRIES);
      *fitting = GetFromSetParameter (jo->file, "job", "fitting", tfit, 1, 0);
      *prtan =
	GetBoolParameter (jo->file, "print", "analytical", PRINT_ANALYTICAL);

      return 1;
    }
  else
    {
      printf ("\nCannot read %s\n", jo->file);
      return 0;
    }
}

//create a new bounds file to pass to gafit
void
SaveBoundsInter (char *file, vect_domain bounds, int types, int ncoefs,
		 int np, int ns)
{
  int i, j;
  FILE *out;

  if ((out = fopen (file, "wt")) != NULL)
    {
      fprintf (out, "internal communication file\n");
      for (j = 0; j < types; j++)
	{
	  for (i = 0; i < ncoefs; i++)
	    {
	      fprintf (out, "\t%+15.5lf  %+15.5lf   %d\n",
		       bounds[j * ncoefs + i].min, bounds[j * ncoefs + i].max,
		       bounds[j * ncoefs + i].decimals);
	    }
	}
      fclose (out);
    }
  else
    {
      printf ("Cannot write file %s\n", file);
      exit (EXIT_FAILURE);
    }
}

vect_domain
ReadJobInternal (JOB * jo, int prt)
{
  char fgeometries[STRING_MAX], fenergies[STRING_MAX], fatypes[STRING_MAX],
    fcharges[STRING_MAX], fbounds[STRING_MAX];

  int np = 0, ns = 0, npt, nst, nct, ndt;
  int pot, types, ncoefs, j, i;
  int rcoef;
  int prtgeo, prtan;
  int fitting;
  int inttypes;			//interaction types: inter, intra+inter(all), user-defined
  int nint;			//user defined interactions
  char *tinter[] = LIST_INTR_TYPE;
  char *bfile;

  if (ReadFIJob
      (jo, fgeometries, fenergies, fatypes, fcharges, fbounds,
       &pot, &rcoef, &prtgeo, &fitting, &prtan, &inttypes, &nint))
    {
      if (prt)
	{
	  printf ("\n");
	  printf ("INTERMOLECULAR MODULE\n");
	  printSepBar (stdout);
	  printf ("Coordinates:[%s]\n", fgeometries);
	  printf ("Energies:[%s]\n", fenergies);
	  printf ("Atom2type:[%s]\n", fatypes);
	  printf ("Bounds:[%s]\n", fbounds);
	  printf ("Charges:[%s]\n", fcharges);
	  printf ("Potential read: ");
	  if (!pot)
	    printf ("Analytical expression\n");
	  else
	    {
	      printf ("%d\n", pot);
	      prtan = 0;
	    }
	  printf ("All coefficients: %s, %s\n", rcoef ? "yes" : "no",
		  ReadCoefs (rcoef));
	  printf ("Interactions types:");
	  printf (" %s", tinter[inttypes]);
	  if (nint)
	    printf (", %d\n", nint);
	  else
	    printf ("\n");
	  printf ("Fitting:");
	  switch (fitting)
	    {
	    case 0:
	      printf (" %s\n", AFIT);
	      break;
	    case 1:
	      printf (" %s\n", RFIT);
	      break;
	    case 2:
	      printf (" %s\n", UFIT);
	      break;
	    default:
	      printf (" unknown\n");
	      exit (EXIT_FAILURE);
	    }
	  printf ("\n");
	  printf ("PRINT OPTIONS\n");
	  printSepBar (stdout);
	  printf
	    ("\tgeometries %s \n\tanalytical %s",
	     prtgeo ? "yes" : "no", prtan ? "yes" : "no");
	  printf ("\n");
	}


      //*********** hook for analytical functions****//    
      if (pot == 0)
	{
	  if (!prt)
	    prtan = 0;
	  if (prtan)
	    {
	      printf ("\n");
	      printf ("Analytical expression\n");
	      printSepBar (stdout);
	    }
	  anCheckAnalytical (jo->file, "prog", prtan);
	}
      //*********************************************//


      // setting printing variables
      if (!prt)
	{
	  prtgeo = 0;
	}

      SetIntrxTypes (&inttypes, &nint);	//set interaction type: inter, intra+inter or user-defined

      SetPrintGeo (&prtgeo);
      // setting fitting
      SetFitting (&fitting);
      // reading fitting data
      Reading (fgeometries, fenergies, fatypes, fcharges, &pot,
	       &ncoefs, strlen (fgeometries), strlen (fenergies),
	       strlen (fatypes), strlen (fcharges));
      if (pot)
	{
	  cnFile (jo->file, ncoefs);
	  // char *letra = CoefName (0);
	}
      else
	{
	  cnFunction (anCoefVar);
	}
      types = GetInteractions ();

      jo->N = ncoefs * types;
      np = GetNProx ();
      ns = GetNSam ();
      npt = GetPTypes ();
      nst = GetSTypes ();
      nct = GetCTypes ();
      ndt = GetDTypes ();
      if (prt)
	{
	  printf ("\n");
	  printf ("INTERACTIONS\n");
	  printSepBar (stdout);
	  printf
	    ("Different interaction types: %d,\n\twith %d coefficients each,\n\tso, we need a %d elements vector.\n\tChoosen potential=%d\n",
	     types, ncoefs, jo->N, pot);

	  printf ("Fragment A atoms: %d", np);
	  if (ns)
	    printf (", Fragment B atoms: %d\n", ns);
	  else
	    printf ("\n");
	  printf ("Fragment A types: %d", npt);
	  if (nst)
	    printf (", Fragment B types: %d\n", nst);
	  else
	    printf ("\n");
	  if (nct)
	    printf ("A and B common types: %d\n", nct);
	  if (ndt)
	    printf ("Interactions between same type: %d\n", ndt);
	}
      vect_domain bounds = malloc (sizeof (domain) * jo->N);
      if (prt)
	{
	  printf ("\n");
	  printf ("Reading bounds for %d coefficients\n", ncoefs);
	  printf ("\n");
	}

      if (!rcoef)
	{
	  vect_domain tbounds = malloc (sizeof (domain) * ncoefs);
	  ReadBounds (tbounds, fbounds, ncoefs, 1);
	  for (i = 0; i < ncoefs; i++)
	    {
	      if (prt)
		{
		  char *clet = CoefName (i);
		  printf ("\t%10s %+15.5lf - %+15.5lf (%s)\n", clet,
			  tbounds[i].min, tbounds[i].max,
			  ii (tbounds[i].decimals));
		}
	    }
	  for (j = 0; j < types; j++)
	    {
	      for (i = 0; i < ncoefs; i++)
		{
		  bounds[j * ncoefs + i].min = tbounds[i].min;
		  bounds[j * ncoefs + i].max = tbounds[i].max;
		  bounds[j * ncoefs + i].gap = tbounds[i].gap;
		  bounds[j * ncoefs + i].decimals = tbounds[i].decimals;
		}
	    }
	  free (tbounds);
	}
      else
	{
	  ReadBounds (bounds, fbounds, ncoefs, types);
	  for (i = 0; i < types * ncoefs; i++)
	    {
	      if (prt)
		{
		  char *clet = CoefName (i % ncoefs);
		  printf ("\t%5d %10s %+15.5lf - %+15.5lf (%s)\n", (i + 1),
			  clet, bounds[i].min,
			  bounds[i].max, ii (bounds[i].decimals));
		  if ((i + 1) % ncoefs == 0)
		    printf ("\n");
		}
	    }
	}
      if (prt)
	{
	  printf ("\n");
	  printf (" %d BOUNDS VECTOR:\n", jo->N);
	  printSepBar (stdout);
	  printf ("\n");
	  PrintBounds (bounds, types, ncoefs, np, ns, 1, types);
	}
      bfile = MakePath (fbounds, ".internal");
      SaveBoundsInter (bfile, bounds, types, ncoefs, np, ns);
      free (bfile);
      return bounds;
    }
  else
    {
      printf ("Cannot open job file\n");
      return NULL;
    }
}
