/*
 (c)GAFit toolkit $Id: fitview.c 510 2025-04-22 14:23:13Z ro $
*/
#if HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include "interface.h"
#include "final.h"
#include "../inputline/line.h"
#include "../analytical/analytical.h"
#include "../literals/literals.h"
#include "../rstrings/rstrings.h"
#include "../cnames.h"
#include "../parameters/parameters.h"

#include "../job.h"


double linf = 0.5;
double lsup = 10.0;
double inc = 0.01;
int enhanced = 0;
char *terminal = NULL;

void
PrintOneType (FILE * outfile, double *x, int a1, int a2, double linf,
	      double lsup, double inc, int N)
{
  double res, dist;
  for (dist = linf; dist < lsup; dist += inc)
    {
      res = Curve (&dist, &a1, &a2, x, &N);
      fprintf (outfile, "\t %+20.10lf   %+20.10lf\n", dist, res);
    }
}

void
PrintCurveInteractions (double *x, int types, int ncoefs, int np, int ns,
			int N)
{
  // some initializations  only for shut up warnings

  FILE *outdat = NULL, *outplt = NULL;
  FILE *aoutplt;
  char *inttype = "2body-type-";
  char *datfile = NULL, *pltfile = NULL, nomfile[STRING_MAX];
  char *apltfile;

  int i, j, k, l, zero, c, bus;
  int charges;
  float qi = 0, qj = 0;
  char at1[10], at2[10];
  charges = NeedQ ();
  int a1, a2, itype = 0;
  int first, pass;

  snprintf (nomfile, STRING_MAX - 1, "%sall", inttype);
  apltfile = MakePath (nomfile, ".plt");

  pass = 1;
  aoutplt = fopen (apltfile, "w");
  if (terminal != NULL)
    fprintf (aoutplt, "set terminal %s\n", terminal);
  else if (enhanced)
    fprintf (aoutplt, "set terminal x11 enhanced\n");
  else
    fprintf (aoutplt, "set terminal x11\n");
  fprintf (aoutplt, "set title \"All interactions\"\n");
  fprintf (aoutplt, "set xrange [%f:%f]\n", linf, lsup);
  fprintf (aoutplt, "set xlabel \"R\"\n");
  fprintf (aoutplt, "set ylabel \"Potential\"\n");
  for (j = 0; j < types; j++)
    {
      for (i = 0; i < ncoefs; i++)
	{
	  if ((j * ncoefs + i) % ncoefs == 0)
	    {
	      first = 1;
	      bus = j * ncoefs + i + 1;
	      itype = bus / ncoefs + 1;
	      snprintf (nomfile, STRING_MAX - 1, "%s%d", inttype, itype);
	      datfile = MakePath (nomfile, ".dat");
	      pltfile = MakePath (nomfile, ".plt");

	      outdat = fopen (datfile, "w");
	      outplt = fopen (pltfile, "w");

	      zero = 1;
	      if (!GetIsUserInteractions ())
		{
		  int nlast, nfirst;
		  c = 0;
		  if (GetIsAllInteractions ())
		    nlast = np + ns;
		  else
		    nlast = np;
		  for (k = 1; k <= nlast; k++)
		    {
		      if (GetIsAllInteractions ())
			nfirst = k + 1;
		      else
			nfirst = np + 1;
		      for (l = nfirst; l <= ns + np; l++)
			{
			  GetAtom (&k, at1);
			  GetAtom (&l, at2);
			  sRtrim (at1);
			  sRtrim (at2);
			  if (IndexType (&k, &l, &zero) == bus)
			    {
			      if (first)
				{
				  a1 = k;
				  a2 = l;
				  first = 0;
				}

			      fprintf (outdat, "#\n");
			      fprintf (outdat, "#INTERACTION TYPE %d\n",
				       itype);
			      fprintf (outdat, "#----------------------\n");
			      fprintf (outdat, "# %s(%d)-%s(%d) ", at1, k,
				       at2, l);

			      c++;
			      if (c % 6 == 0)
				fprintf (outdat, "\n#");
			      if (charges)
				{
				  qi = GetQ (&k);
				  qj = GetQ (&l);
				}
			    }
			}
		    }
		  if (charges)
		    fprintf (outdat, "\n#\n#\tqi=%+15.5lf\n\tqj=%+15.5lf\n",
			     qi, qj);
		}
	      fprintf (outdat, "\n#\tCoefficients:\n");
	    }
	  if (GetPot ())
	    {
	      fprintf (outdat, "#\t%5d %s %+20.10lf\n",
		       j * ncoefs + i + 1,
		       CoefName ((j * ncoefs + i) % ncoefs),
		       x[j * ncoefs + i]);
	    }
	  else
	    fprintf (outdat, "#\t%5d %10s %+20.10lf\n",
		     j * ncoefs + i + 1,
		     CoefName ((j * ncoefs + i) % ncoefs), x[j * ncoefs + i]);

	}
      fprintf (outdat, "#\n#\t\t\tr\t\tV\n");
      PrintOneType (outdat, x, a1, a2, linf, lsup, inc, N);
      fprintf (outdat, "\n");
      if (terminal != NULL)
	fprintf (outplt, "set terminal %s\n", terminal);
      else if (enhanced)
	fprintf (outplt, "set terminal x11 enhanced\n");
      else
	fprintf (outplt, "set terminal x11\n");
      fprintf (outplt, "set title \"Interaction type %d\"\n", itype);
      fprintf (outplt, "set xrange [%f:%f]\n", linf, lsup);
      fprintf (outplt, "set xlabel \"R\"\n");
      fprintf (outplt, "set ylabel \"Potential\"\n");
      GetAtom (&a1, at1);
      GetAtom (&a2, at2);
      if (enhanced)
	fprintf (outplt,
		 "plot \"%s\" using 1:2 title \"Ex: %s_{%d}-%s_{%d}\" with linespoints\n",
		 datfile, at1, a1, at2, a2);
      else
	fprintf (outplt,
		 "plot \"%s\" using 1:2 title \"Ex: %s(%d)-%s(%d)\" with linespoints\n",
		 datfile, at1, a1, at2, a2);

      if (pass)
	fprintf (aoutplt, "plot ");
      else
	fprintf (aoutplt, "replot ");

      if (enhanced)
	fprintf (aoutplt,
		 " \"%s\" using 1:2 title \"%d: %s_{%d}-%s_{%d}\" with linespoints\n",
		 datfile, itype, at1, a1, at2, a2);
      else
	fprintf (aoutplt,
		 " \"%s\" using 1:2 title \"%d: %s(%d)-%s(%d)\" with linespoints\n",
		 datfile, itype, at1, a1, at2, a2);
      pass = 0;

      fprintf (outplt, "pause -1\n");
      fclose (outdat);
      fclose (outplt);
      free (datfile);
      free (pltfile);
    }
  fprintf (aoutplt, "pause -1\n");
  fclose (aoutplt);
  free (apltfile);
}


void
PrintCurves (int N)
{

  double *x;
  int ndi, ncoefs, np, ns;
  ndi = GetInteractions ();
  ncoefs = GetNCoefs ();
  np = GetNProx ();
  ns = GetNSam ();

  x = LoadBest (OUTPUT_BEST, N);

  PrintCurveInteractions (x, ndi, ncoefs, np, ns, N);
  free (x);
}

void
PrintGeneralEvaluation (int N)
{
  FILE *outdat, *outplt;
  char *general = "general_evaluation";
  char *datfile, *pltfile;

  datfile = MakePath (general, ".dat");
  pltfile = MakePath (general, ".plt");

  outdat = fopen (datfile, "w");
  outplt = fopen (pltfile, "w");

  FinalEvaluation (outdat, N, 0, NULL);

  fprintf (outplt, "set title \"General Evaluation\"\n");
  fprintf (outplt, "set xrange [0:%d]\n", GetGeom () + 1);
  fprintf (outplt, "set xlabel \"Geometry number\"\n");
  fprintf (outplt, "set ylabel \"Energy\"\n");
  fprintf (outplt,
	   "plot \"%s\" using 1:2 title \"Ab initio\" with linespoints, ",
	   datfile);
  fprintf (outplt,
	   " \"%s\" using 1:3 title \"Calculated\" with linespoints\n",
	   datfile);
  fprintf (outplt, "pause -1\n");
  fclose (outdat);
  fclose (outplt);
  free (datfile);
  free (pltfile);
}


double
chkoption (char *string)
{
  char *end;
  errno = 0;
  double n = strtod (string, &end);
  if (n == 0)
    {
      if (string == end)
	perror ("cannot parse option\n");
      exit (EXIT_FAILURE);
    }
  if (errno == ERANGE)
    {
      perror ("out of range\n");
      exit (EXIT_FAILURE);
    }
  return n;
}

void
Usage (void)
{
  printf ("fitview v0.4 (c)GAFit toolkit - 2010-2024\n");
  printf ("Usage: fitview [tag] [-l value] [-u value] [-d value] [-h] \n");
  printf ("\t -l lower bound\n");
  printf ("\t -u upper bound\n");
  printf ("\t -d delta\n");
  printf ("\t -t terminal\n");
  printf ("\t -e gnuplot supports enhanced terminal\n");
  printf ("\t -h this help\n");
  printf ("\t -g general evaluation only\n");
  printf ("\t default [%lf,%lf] delta: %lf\n", linf, lsup, inc);
  printf ("\t default terminal 'x11', MacOS try 'aqua'\n");

}

void
ReadJType (JOB * jo)
{
  char *jtype[] = LIST_JOB_TYPES;

  jo->jobtype =
    GetFromSetParameter (jo->file, "job", "type", jtype, JOB_TYPE_SIMPLE, 0);
  GetStringParameter (jo->file, "job", "external input", "external.input",
		      jo->external_input, STRING_MAX);
  GetStringParameter (jo->file, "job", "external fit", "external.fit",
		      jo->external_fit, STRING_MAX);
  GetStringParameter (jo->file, "job", "bounds", "bounds.txt",
		      jo->external_bounds, STRING_MAX);
  jo->external_coefs = GetIntParameter (jo->file, "job", "coefficients", 0);
  jo->final_evaluation =
    GetBoolParameter (jo->file, "job", "final evaluation", 0);
  cnFile (jo->file, jo->external_coefs);
  if (jo->external_coefs < 1)
    {
      printf ("Coefficients must be set\n");
      exit (EXIT_FAILURE);
    }
}


// main function
int
main (int argc, char **argv)
{
  JOB job;
  int onlyGeneral = 0;
  int ta = argc - 1;

  initJob (&job);

  if (ta)
    {
      while (argc--)
	{
	  if (strcmp (argv[argc], "-l") == 0 && argc < ta)
	    {
	      linf = chkoption (argv[argc + 1]);
	      printf ("Lower bound: %lf\n", linf);
	    }

	  else if (strcmp (argv[argc], "-u") == 0 && argc < ta)
	    {
	      lsup = chkoption (argv[argc + 1]);
	      printf ("Upper bound: %lf\n", lsup);
	    }

	  else if (strcmp (argv[argc], "-d") == 0 && argc < ta)
	    {
	      inc = chkoption (argv[argc + 1]);
	      printf ("Delta: %lf\n", inc);
	    }
	  else if (strcmp (argv[argc], "-e") == 0)
	    {
	      enhanced = 1;

	    }
	  else if (strcmp (argv[argc], "-t") == 0)
	    {
	      terminal = strdup (argv[argc + 1]);
	      printf ("terminal=%s\n", terminal);
	    }
	  else if (strcmp (argv[argc], "-g") == 0)
	    {
	      onlyGeneral = 1;
	    }
	  else if (strcmp (argv[argc], "-h") == 0)
	    {
	      Usage ();
	      exit (EXIT_SUCCESS);

	    }
	}
    }
  if (linf >= lsup)
    {
      Usage ();
      exit (EXIT_FAILURE);
    }


  job.bounds = ReadJobInternal (&job, 0);

  if (job.bounds != NULL)
    {
      //print plots
      PrintGeneralEvaluation (job.N);
      if (!onlyGeneral)
	PrintCurves (job.N);
    }
  else
    {
      printf ("\nEND\n");
    }
  return 0;
}
