/*
(c) GAFit toolkitk $Id: grunner.c 558 2025-11-30 01:00:26Z ro $
*/
#if HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <float.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "../flyctl/flyctl.h"
#include "../inputline/line.h"
#include "../rangecf/rangecf.h"
#include "../literals/literals.h"
#include "../environ/environ.h"
#include "../nullist/nllist.h"
#include "../rstrings/rstrings.h"
#include "common.h"


extern char *ExternalInput;
extern char *ExternalFit;
extern char *InputTemplate;
extern int ParallelCalcs;
extern char *ExternalExecutable;
extern char *ReferenceValues;

extern int debug;

#define BULLETBUFFER 2048

void
usage (void)
{
  printf ("genrunner v0.1 (c)GAFit toolkit - 2018\n");
  printf ("\tUsage: genrunner n-of-vectors\n\n");
  exit (EXIT_SUCCESS);
}

double *
WriteTextFit (char *rawfits, char *fits, int indv)
{
  FILE *r, *w;
  int i;
  int fail = 0;
  double *dfit = malloc (sizeof (double) * indv);
  if ((r = fopen (rawfits, "r")) != NULL)
    {

      if ((w = fopen (fits, "w")) != NULL)
	{
	  if (fread (dfit, sizeof (double), indv, r) == indv)
	    {
	      for (i = 0; i < indv; i++)
		{
		  if (dfit[i] == DBL_MAX / INT_MAX)
		    fail++;
		  fprintf (w, "%.16g\n", dfit[i]);
		}
	    }
	  else
	    {
	      fcPrintfStopIt ("read less than %d fits from %s\n", indv,
			      rawfits);
	    }
	  fclose (w);
	}
      else
	{
	  fcPrintfStopIt ("Cannot open %s\n", fits);
	}
      fclose (r);
    }
  else
    {
      fcPrintfStopIt ("Cannot open %s\n", rawfits);
    }
//  free (dfit);
  if (fail > 0)
    printf ("%d failures\n", fail);
  return dfit;
}

/**
 * 	obtains the coefficient's whole set.
 * 	@parameter nindv number of individues
 * 	@parameter rc Range-Coefs struct from template analysis.
 *	@return  all coefficient's sets
 * */
double **
getAllCoefs (int nindv, RANGECOEFS *rc)
{
  char *s;
  double *coefs;
  static double **allCoefs = NULL;

  if (allCoefs != NULL)
    return allCoefs;
  FILE *e = fopen (ExternalInput, "r");
  if (e)
    {
      allCoefs = (double **) malloc (sizeof (double *) * nindv);

      for (int i = 0; i < nindv; i++)
	{
	  coefs = (double *) malloc (sizeof (double) * rc->ncoefs);
	  for (int j = 0; j < rc->ncoefs; j++)
	    {
	      while (isBlankLine (s = InputFileLine (e)));
	      //printf ("i=%d coef=%d total=%d s=%s\n", i, j, rc->ncoefs, s);

	      if (!s)
		{
		  printf ("Error External Input!\n");
		  fcStopIt ("genrunner: No input in file ExternalInput\n");
		}
	      sscanf (s, "%lf", &coefs[j]);
	      free (s);
	    }
	  allCoefs[i] = coefs;
	}
      return allCoefs;
    }
  return NULL;
}

void
GenerateInputFiles (int nindv, RANGECOEFS *rc, char *templateName)
{
  FILE *e;
  int i;
  char *text, *text2;
  char *file;
  double **allCoefs;
  double *coefs;
  e = fopen (templateName, "r");
  if (e)
    {
      text = InputFileText (e);
      fclose (e);

      if ((allCoefs = getAllCoefs (nindv, rc)) != NULL)
	{
	  for (i = 0; i < nindv; i++)
	    {
	      coefs = allCoefs[i];
	      text2 = strdup (text);
	      text2 = rcCreateParameters (text2, coefs, rc);
	      if (text2 == NULL)
		fcStopIt ("Error rcCreateParameters\n");

	      file = Letters (i);
	      WriteInputFile (text2, file, templateName);
	      free (text2);
	      free (file);
	    }
	  free (text);
	}
      return;
    }
  fcStopIt ("generateInpuFiles error\n");
}

void
argError (int n)
{
  if (n < 1)
    {
      printf ("Argument error:%d\n", n);
      fcStopIt ("genrunner: argument error\n");
    }
}


int
bulletError (char *output)
{
  if (strstr (output, "error"))
    return 1;
  return 0;
}

int
bullet (char *executable, char *inputfile)
{
  pid_t bulletPid;
  int errpipe[2];
  int ostderr;
  int ret = 0;
  char input[BULLETBUFFER];
  int rbytes;
  char *arguments[3];

  arguments[2] = NULL;
  arguments[1] = inputfile;
  arguments[0] = executable;

  if (!pipe (errpipe))
    {
      ostderr = dup (2);
      close (2);
      dup2 (errpipe[1], 2);
      //second fork
      if (!(bulletPid = fork ()))
	{
	  //worker child
	  close (errpipe[0]);
	  close (errpipe[1]);
	  execv (arguments[0], arguments);
	  //returns here if fail  
	  perror (arguments[0]);
	  fcPrintfStopIt ("execv %s %s:  %s\n", arguments[0], inputfile,
			  strerror (errno));
	}
      else
	{
	  //controling parent
	  close (2);

	  dup2 (ostderr, 2);

	  close (errpipe[1]);

	  //reading from child stderr: 
	  if ((rbytes = read (errpipe[0], input, BULLETBUFFER)) > 0)
	    {
	      input[rbytes] = 0;	//zero terminated string
	      //filter here err msg *************************
	      if (bulletError (input))
		{
		  printf ("grunner bullet %s errno %d  %s", inputfile,
			  errno, input);
		  //*********************************************
		  //let child write __STOP__ file 
		  sleep (1);
		  //check for __STOP__ file
		  fcCheck ();
		  kill (bulletPid, SIGKILL);
		  ret = 1;
		}
	    }
	  //barrier .... if overflows buffer
	  waitpid (bulletPid, NULL, 0);
	}
      return ret;
    }
  else
    {
      fcStopIt ("bullet: pipe call doesn't work!\n");
    }
  return 1;
}


void
forker (char *cmd, char *file)
{
  //first fork main program continue running...
  pid_t pid = fork ();

  if (pid == 0)
    {
      //child 
      if (!bullet (cmd, file))
	exit (EXIT_SUCCESS);
      else
	exit (EXIT_FAILURE);
    }
  //main program...
  if (pid == -1)
    {
      fcPrintfStopIt ("grunner fork failed: %s %s\n", cmd, file);
    }
}

void
ScriptMachineGun (int n, int parallel, char *cmd)
{
  char *file;
  int ifile = 0;

  //less calcs than permited parallel jobs
  if (n < parallel)
    parallel = n;
  //fill up first parallel jobs 
  for (int i = 0; i < parallel; i++)
    {
      file = Letters (ifile);
      ifile++;
      forker (cmd, file);
      free (file);
    }
  fcCheck ();
  while (1)
    {
      int status;
      pid_t done = wait (&status);	//wait child termination
      if (done == -1)
	{
	  if (errno == ECHILD)
	    break;		// no more child processes
	}
      else
	{
	  if (!WIFEXITED (status) || WEXITSTATUS (status) != 0)
	    {
	      fcPrintfStopIt ("pid %d failed\n", done);
	    }
	  //one more
	  if (ifile < n)
	    {
	      file = Letters (ifile);
	      ifile++;
	      forker (cmd, file);
	      free (file);
	    }
	}
      fcCheck ();
    }
}

char *
xRaw (int indv, char *name, int ndoubles)
{
  FILE *f;
  double *d = malloc (sizeof (double) * ndoubles);

  char *rawfile = getcwd (NULL, 0);

  for (int i = 0; i < ndoubles; i++)
    d[i] = DBL_MAX / INT_MAX;	//max finite double number/ max positive integer value

  rawfile = realloc (rawfile, strlen (rawfile) + strlen (name) + 2);
  rawfile = strcat (rawfile, "/");
  rawfile = strcat (rawfile, name);
  unlink (rawfile);

  if ((f = fopen (rawfile, "w")) != NULL)
    {

      for (int i = 0; i < indv; i++)
	fwrite (d, ndoubles, 1, f);

      fclose (f);

      free (d);

      return rawfile;
    }
  fcPrintfStopIt ("cannot read rawfile:%s\n", rawfile);
  return NULL;
}

char *
RawFits (int indv)
{
  return xRaw (indv, RAW_FITS, 1);
}


char *
RawResults (int indv, int nvalues)
{
  return xRaw (indv, RAW_RESULTS, nvalues);
}

char *
RefValues (void)
{
  char *ref = getcwd (NULL, 0);

  ref = realloc (ref, strlen (ref) + strlen (ReferenceValues) + 2);
  ref = strcat (ref, "/");
  ref = strcat (ref, ReferenceValues);

  return ref;
}

int
main (int argc, char **argv)
{
  int indv, nvalues;
  char *rawFitsFile;
  char *refValuesFile;
  char *rawResultsFile;
  char *RawIntermediate;

  double *fits, *results;

  RANGECOEFS *rc;
  REFERENCEVALUES **referenceValuesInfo;
  ALLRESULTS *all = NULL;

  rc = rcFactory ();
  fcStartFly ();

  unlink ("response");

  InitEnvironmentVariables ();

  if (strcmp (ExternalExecutable, EE_NOT_SET) == 0)
    {
      fcStopIt (ExternalExecutable);
    }

  if (argc == 1)
    {
      usage ();
    }
  else if (argc > 2)
    {
      printf ("Error: number of arguments!\n");
      fcStopIt ("grunner: number of arguments\n");
    }
  else
    {
      char **tNames = SplitTemplates (InputTemplate);
      int n = nllCount (tNames);
      double **coefficients;

      indv = atoi (argv[1]);
      argError (indv);

      if ((rc = rcReadAnalysisResults (ANALYSIS_RESULTS_FILE)) == NULL)
	{
	  fcStopIt ("Error readAnalysisResults\n");
	}

      coefficients = getAllCoefs (indv, rc);

      //loop to generate input files
      for (int i = 0; i < n; i++)
	{
	  GenerateInputFiles (indv, rc, nllString (tNames, i));
	}
      //end loop
      nllClear (tNames);

      //sets file names
      rawFitsFile = RawFits (indv);

      setenv ("RAW_FITS_FILE", rawFitsFile, 1);

      refValuesFile = RefValues ();

      setenv ("REFERENCE_VALUES_FILE", refValuesFile, 1);

      referenceValuesInfo = getNRefs (refValuesFile, &nvalues);

      rawResultsFile = RawResults (indv, nvalues);

      setenv ("RAW_RESULTS", rawResultsFile, 1);

      RawIntermediate = RAW_INTERMEDIATE_RESULTS;

      //parallel execution
      ScriptMachineGun (indv, ParallelCalcs, ExternalExecutable);

      fits = WriteTextFit (rawFitsFile, ExternalFit, indv);	//nindv double*

      results = ReadResults (rawResultsFile, indv, nvalues);	//nindv * nvalues double*

      all =
	loadNewResults (indv, nvalues, rc, coefficients, results, fits,
			referenceValuesInfo);
      free (coefficients);
      //printf ("new results\n");
      //printAllResults (stdout, all, YESDUPS, NOLIMIT);

      if (FileSize (RawIntermediate))
	{
	  all = ReadIntermediateResults (RawIntermediate, all);	//read last execution combined results, references and fits
//        printf ("from file intermediates\n");
//        printAllResults (stdout, all, YESDUPS, NOLIMIT);

	}
      //NODUPS NOLIMIT

      //printf ("added intermediates\n");
      all = calcDist (all);
      all = sortAllD (all);
      //    printAllResults (stdout, all, YESDUPS, NOLIMIT);
      all = applyPercentile (all);	// apply centiles.
      //     printf ("after percentile\n");
      //     printAllResults (stdout, all, YESDUPS, NOLIMIT);
      WriteTextResults (ALL_TEXT_RESULTS, all);
      WriteIntermediateResults (RawIntermediate, all);
      free (rawFitsFile);
      free (rawResultsFile);
      free (refValuesFile);
      for (int i = 0; i < all->nresults; i++)
	{
	  free (all->results[i]->coefs);
	  free (all->results[i]->values);
	}
      free (all->results);
      free (all);
      free (fits);
      free (referenceValuesInfo);
      free (results);
      exit (EXIT_SUCCESS);
    }

  exit (EXIT_FAILURE);
}
