/*
(c) GAFit toolkit $Id: fpu.c 510 2025-04-22 14:23:13Z ro $
*/
#if HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include <limits.h>
#include <errno.h>
#include <math.h>
#include <fenv.h>
#include "../inputline/line.h"
#include "../pack/pack.h"
#include "fpu.h"
#include "../bytecodes/bytecodes.h"
#ifdef __DEBUG__
#define DEBUGPRINT(...)  printf( __VA_ARGS__)
#else
#define DEBUGPRINT(...)		//printf( __VA_ARGS__)
#endif
void
fpuPushAddrStack (FPU_RUNTIME * r, double *addr)
{
  r->AddrStack =
    (double **) realloc (r->AddrStack,
			 sizeof (double *) * (r->StackPointer + 1));
  if (!r->AddrStack)
    {
      printf ("fpuPushAddrStack realloc failed\n");
      exit (EXIT_FAILURE);
    }
  r->AddrStack[r->StackPointer] = addr;
  r->StackPointer++;
}

void
fpuPopAddrStack (FPU_RUNTIME * r)
{
  r->StackPointer--;
  if (r->StackPointer > 0)
    {
      r->AddrStack =
	(double **) realloc (r->AddrStack,
			     sizeof (double *) * (r->StackPointer));
    }
  else
    {
      free (r->AddrStack);
      r->AddrStack = NULL;
      if (r->StackPointer < 0)
	r->FpuFlags |= FPU_STACKEMPTY;
      r->StackPointer = 0;
    }
}

int
ifs (int s)
{
  if (s)
    return 1;
  return 0;
}

void
fpuStatus (int flags)
{
  printf ("status[%d|%d|%d|%d|%d|%d|%d|%d|%d|%d]",
	  ifs (flags & FPU_ERROR),
	  ifs (flags & FPU_STACKEMPTY),
	  ifs (flags & FPU_BADADDR),
	  ifs (flags & FPU_NOTINT),
	  ifs (flags & FPU_INSTRERROR),
	  ifs (flags & FPU_MATHERR),
	  ifs (flags & FPU_DIVBYZERO),
	  ifs (flags & FPU_INVALID),
	  ifs (flags & FPU_UNDERFLOW), ifs (flags & FPU_OVERFLOW));
}

void
fpuExplainStatus (int flags)
{
  if (ifs (flags & FPU_ERROR))
    printf ("Decode error:");
  if (ifs (flags & FPU_STACKEMPTY))
    printf ("Stack exhausted\n");
  if (ifs (flags & FPU_BADADDR))
    printf ("Bad address\n");
  if (ifs (flags & FPU_NOTINT))
    printf ("Needed an int to address memory pool\n");
  if (ifs (flags & FPU_INSTRERROR))
    printf ("Instruction unknown\n");
  if (ifs (flags & FPU_MATHERR))
    printf ("Math error:");
  if (ifs (flags & FPU_DIVBYZERO))
    printf ("Divide by zero\n");
  if (ifs (flags & FPU_INVALID))
    printf ("Invalid operation 0/0 or such\n");
  if (ifs (flags & FPU_UNDERFLOW))
    printf ("Underflow\n");
  if (ifs (flags & FPU_OVERFLOW))
    printf ("Overflow\n");
}

void
fpuPrintAddrStack (FPU_RUNTIME * r)
{
  int i;

  printf ("\n\t");
  fpuStatus (r->FpuFlags);
  printf ("\n");
  printf ("\t[Stack pointer %d ", r->StackPointer);
  for (i = 0; i < r->StackPointer; i++)
    {
      printf (" %d=", i);
      printf ("{%lg}", *r->AddrStack[i]);
    }
  printf ("]\n");
}

void
fpuPrintMemory (FPU2011 * u)
{
  int i;
  printf ("\t[Mem counter: %d: ", u->MemoryCount);
  for (i = 0; i < u->MemoryCount; i++)
    {
      printf (" %d=", i);
      printf ("{%lg}", *u->MemoryPool[i]);
    }
  printf ("]\n");
}

void
fpuPrintMemoryLabels (FPU2011 * u, char **vars, int nvars)
{
  int i;
  int min;
  min = nvars > u->MemoryCount ? u->MemoryCount : nvars;
  printf ("\tMemory (total used %d) ", u->MemoryCount);
  for (i = 0; i < min; i++)
    {
      printf (" %s=", *vars);
      printf ("%lg", *u->MemoryPool[i]);
      vars++;
    }
  printf ("\n");
}

void
fpuMemoryPoolAdd (FPU2011 * u)
{
  u->MemoryPool =
    (double **) realloc (u->MemoryPool,
			 sizeof (double *) * (u->MemoryCount + 1));
  u->MemoryPool[u->MemoryCount] = (double *) malloc (sizeof (double));
  u->MemoryCount++;
}

void
clearUnderFlow (void)
{
  if (fetestexcept (FE_UNDERFLOW))
    {
      feclearexcept (FE_UNDERFLOW);
    }
}

int
fpu2011 (FPU2011 * run)
{
  FPU_RUNTIME r;

  r.AddrStack = NULL;
  r.StackPointer = 0;
  r.ProgramCounter = 0;
  r.FpuFlags = 0;


  enum BYTECODE instruction;
  void *buffer = packBuff ();
  enum PACKTYPE type = 0;
  DEBUGPRINT ("Program decode:\n");
  if (run->Program)
    {
#ifdef __DEBUG__
      printPack (run->Program);
#endif
      r.ProgramCounter = 0;
      do
	{
	  type = upack (run->Program, &r.ProgramCounter, buffer);
	  DEBUGPRINT ("Decode %d ", type);
	  switch (type)
	    {
	    case BYTECODE:
	      DEBUGPRINT ("bytecode ");
	      instruction = *(enum BYTECODE *) buffer;
	      DEBUGPRINT ("instruction %s ", byteCodeString (instruction));
	      switch (instruction)
		{
		case PUSH:
		  type = upack (run->Program, &r.ProgramCounter, buffer);
		  DEBUGPRINT ("type %d data\n", type);
		  fpuMemoryPoolAdd (run);
		  fpuPushAddrStack (&r,
				    run->MemoryPool[run->MemoryCount - 1]);
		  *run->MemoryPool[run->MemoryCount - 1] = *(double *) buffer;
		  break;
		case POP:
		  fpuPopAddrStack (&r);
		  break;
		case STORE:
		  if (r.StackPointer - 2 < 0)
		    r.FpuFlags |= FPU_STACKEMPTY;
		  else
		    {
		      *r.AddrStack[r.StackPointer - 2] =
			*r.AddrStack[r.StackPointer - 1];
		      //pop 1
		      fpuPopAddrStack (&r);
		      //pop 2
		      fpuPopAddrStack (&r);
		    }
		  break;
		case MOVE:
		  type = upack (run->Program, &r.ProgramCounter, buffer);
		  if (type != INT)
		    r.FpuFlags |= FPU_NOTINT;
		  else
		    {

		      DEBUGPRINT ("into %d\n", *(int *) buffer);
		      *run->MemoryPool[*(int *) buffer] =
			*r.AddrStack[r.StackPointer - 1];
		    }
		  break;
		case APUSH:
		  type = upack (run->Program, &r.ProgramCounter, buffer);
		  if (type != INT)
		    r.FpuFlags |= FPU_NOTINT;
		  else if (!run->MemoryPool)
		    r.FpuFlags |= FPU_BADADDR;
		  else
		    {
		      int a = *(int *) buffer;
		      DEBUGPRINT ("memory %d\n", a);
		      if (a > run->MemoryCount - 1)
			{
			  r.FpuFlags |= FPU_BADADDR;
			}
		      else
			{
			  fpuPushAddrStack (&r, run->MemoryPool[a]);
			}
		    }
		  break;
		case CLRF:
		  DEBUGPRINT ("clrf\n");
		  r.FpuFlags = 0;
		  break;
		case ADD:
		case SUB:
		case MULT:
		case DIV:
		case POW:
		  if (r.StackPointer - 2 < 0)
		    {
		      r.FpuFlags |= FPU_STACKEMPTY;
		      DEBUGPRINT ("\n");
		    }
		  else
		    {
		      double a, b;
		      type = DOUBLE;
		      a = *r.AddrStack[r.StackPointer - 1];
		      b = *r.AddrStack[r.StackPointer - 2];

		      feclearexcept (FE_ALL_EXCEPT);

		      switch (instruction)
			{
			case ADD:
			  b += a;
			  break;
			case SUB:
			  b -= a;
			  break;
			case MULT:
			  b *= a;
			  clearUnderFlow ();
			  break;
			case DIV:
			  if (a == 0)
			    {
			      r.FpuFlags |= FPU_DIVBYZERO | FPU_MATHERR;
			      break;
			    }
			  b /= a;
			  break;
			case POW:
			  b = pow (b, a);
			  break;
			default:
			  printf ("instruction %d error\n", instruction);
			  exit (EXIT_FAILURE);
			}

		      //catch...1
		      if (fetestexcept
			  (FE_INVALID | FE_DIVBYZERO | FE_OVERFLOW |
			   FE_UNDERFLOW))
			{
			  r.FpuFlags |= FPU_MATHERR;
			  if (fetestexcept (FE_DIVBYZERO))
			    r.FpuFlags |= FPU_DIVBYZERO;
			  if (fetestexcept (FE_INVALID))
			    r.FpuFlags |= FPU_INVALID;
			  if (fetestexcept (FE_OVERFLOW))
			    r.FpuFlags |= FPU_OVERFLOW;
			  if (fetestexcept (FE_UNDERFLOW))
			    r.FpuFlags |= FPU_UNDERFLOW;
			  break;
			}


		      fpuMemoryPoolAdd (run);
		      *run->MemoryPool[run->MemoryCount - 1] = b;
		      //pop 1
		      fpuPopAddrStack (&r);
		      //pop 2
		      fpuPopAddrStack (&r);

		      fpuPushAddrStack (&r,
					run->MemoryPool[run->MemoryCount -
							1]);
		    }
		  break;
		case EXP:
		case SIN:
		case COS:
		case NEG:
		  if (r.StackPointer - 1 < 0)
		    r.FpuFlags |= FPU_STACKEMPTY;
		  else
		    {
		      double a = *r.AddrStack[r.StackPointer - 1];
		      feclearexcept (FE_ALL_EXCEPT);
		      switch (instruction)
			{
			case SQRT:
			  if (a < 0.0)
			    {
			      r.FpuFlags |= FPU_INVALID | FPU_MATHERR;
			      break;
			    }
			  a = sqrt (a);
			  break;
			case EXP:
			  a = exp (a);
			  clearUnderFlow ();
			  break;
			case SIN:
			  a = sin (a);
			  break;
			case COS:
			  a = cos (a);
			  break;
			case NEG:
			  a = -a;
			  break;
			default:
			  printf ("2.-instruction %d error\n", instruction);
			  exit (EXIT_FAILURE);
			}

		      //catch...2
		      if (fetestexcept
			  (FE_INVALID | FE_DIVBYZERO | FE_OVERFLOW |
			   FE_UNDERFLOW))
			{
			  r.FpuFlags |= FPU_MATHERR;
			  if (fetestexcept (FE_DIVBYZERO))
			    r.FpuFlags |= FPU_DIVBYZERO;
			  if (fetestexcept (FE_INVALID))
			    r.FpuFlags |= FPU_INVALID;
			  if (fetestexcept (FE_OVERFLOW))
			    r.FpuFlags |= FPU_OVERFLOW;
			  if (fetestexcept (FE_UNDERFLOW))
			    r.FpuFlags |= FPU_UNDERFLOW;
			  printf ("\n\t error catched!\n\tflags:");
			  fpuStatus (r.FpuFlags);
			  printf ("\n\tinstruction:%s operand:%lg\n",
				  byteCodeString (instruction), a);
			  fpuExplainStatus (r.FpuFlags);
			  break;
			}


		      fpuMemoryPoolAdd (run);
		      *run->MemoryPool[run->MemoryCount - 1] = a;
		      fpuPopAddrStack (&r);
		      fpuPushAddrStack (&r,
					run->MemoryPool[run->MemoryCount -
							1]);
		    }
		  break;
		default:
		  printf ("3-instruction %d error\n", instruction);
		  exit (EXIT_FAILURE);
		}
	      break;
	    case END:
	      DEBUGPRINT ("END\n");
	      break;
	    default:
	      DEBUGPRINT ("type %d error\n", type);
	      r.FpuFlags |= FPU_INSTRERROR;
	      break;
	    }
	  if (r.FpuFlags & FPU_ERROR || r.FpuFlags & FPU_MATHERR)
	    break;
#ifdef __DEBUG__
	  fpuPrintAddrStack (&r);
	  fpuPrintMemory (run);
#endif
	}
      while (type != END);
      if (r.StackPointer > 0)
	{
	  run->StackTop = r.AddrStack[r.StackPointer - 1];
	}
      else
	{
	  run->StackTop = NULL;
	}
      free (r.AddrStack);
      free (buffer);
    }
  return r.FpuFlags;
}

//passing a contigous memory block!!
void
fpuMemoryBlockAdd (FPU2011 * u, double *mblock, int size)
{
  int i;
  u->MemoryPool =
    (double **) realloc (u->MemoryPool, sizeof (double *) * size);
  for (i = 0; i < size; i++)
    u->MemoryPool[i] = mblock + i;
  u->MemoryCount = size;
  u->MemoryProtect = size;
}

//passing a variable
void
fpuAddVariable (FPU2011 * u, double variable)
{
  fpuMemoryPoolAdd (u);
  *u->MemoryPool[u->MemoryCount - 1] = variable;
  u->MemoryProtect++;
}

//getting a variable
double
fpuGetVariable (FPU2011 * u, int index)
{
  return *u->MemoryPool[index];
}

void
fpuReset (FPU2011 * fpu)
{
  int i = fpu->MemoryProtect;
  while (i < fpu->MemoryCount)
    free (fpu->MemoryPool[i++]);
  free (fpu->MemoryPool);
  fpu->MemoryPool = NULL;
  fpu->MemoryCount = 0;
  fpu->MemoryProtect = 0;
}

void
fpuClear (FPU2011 * fpu)
{
  fpuReset (fpu);
  free (fpu->Program);
}

void
fpuLoad (FPU2011 * fpu, char *file)
{
  int fsize = (int) FileSize (file);
  FILE *f;
  fpu->Program = (void *) malloc (fsize);
  f = fopen (file, "rb");
  fread (fpu->Program, fsize, 1, f);
  fclose (f);
}

FPU2011 *
fpuNew (void)
{
  FPU2011 *fpu = (FPU2011 *) malloc (sizeof (FPU2011));
  fpu->Program = NULL;
  fpu->ProgramLen = 0;
  fpu->MemoryPool = NULL;
  fpu->MemoryCount = 0;
  fpu->MemoryProtect = 0;
  fpu->StackTop = NULL;
  return fpu;
}
