#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <math.h>
#include <stdlib.h>
#include <time.h>

#include "core.h"
#include "job.h"
#include "numbers.h"
#include "rand.h"
#include "crossover.h"

/* Crossover Types

 	0: single point crossover
	1: double point crossover
	2: BLX-alpha crossover
  	3: SBX (Simulated Binary Crossover - proposed by Deb)
 	
*/


void
single_point_crossover (int ind_size, double p1[], double p2[], double off1[],
			double off2[])
{
  int i, position;

  position = random_number (1, ind_size - 2);
  for (i = 0; i < position; i++)
    {
      off1[i] = p1[i];
      off2[i] = p2[i];
    }

  for (i = position; i < ind_size; i++)
    {
      off1[i] = p2[i];
      off2[i] = p1[i];
    }
}


void
double_point_crossover (int ind_size, double p1[], double p2[], double off1[],
			double off2[])
{
  int i, position1, position2, temp;

  position1 = random_number (1, ind_size - 2);
  temp = random_number (1, ind_size - 2);

  if (position1 < temp)
    {
      position2 = temp;
    }
  else
    {
      position2 = position1;
      position1 = temp;
    }

  for (i = 0; i < position1; i++)
    {
      off1[i] = p1[i];
      off2[i] = p2[i];
    }
  for (i = position1; i < position2; i++)
    {
      off1[i] = p2[i];
      off2[i] = p1[i];
    }
  for (i = position2; i < ind_size; i++)
    {
      off1[i] = p1[i];
      off2[i] = p2[i];
    }
}


void
SBX_crossover (int ind_size, double p1[], double p2[], double off1[],
	       double off2[], double eta_sbx, vect_domain bounds)
{
  int i;
  double mu, beta;

  do
    {
      mu = uniform01 ();
    }
  while (mu > 0.99);

  if (mu <= 0.5)
    {
      beta = pow (2 * mu, (1.0 / (eta_sbx + 1.0)));
    }
  else
    {
      beta = pow (1.0 / (2.0 * (1 - mu)), (1.0 / (eta_sbx + 1.0)));
    }

  for (i = 0; i < ind_size; i++)
    {
      if (!isInt (bounds[i].decimals))
	{
	  off1[i] = 0.5 * ((1.0 + beta) * p1[i] + (1.0 - beta) * p2[i]);
	  if (off1[i] < bounds[i].min)
	    off1[i] = bounds[i].min;
	  else if (off1[i] > bounds[i].max)
	    off1[i] = bounds[i].max;

	  off2[i] = 0.5 * ((1.0 - beta) * p1[i] + (1.0 + beta) * p2[i]);
	  if (off2[i] < bounds[i].min)
	    off2[i] = bounds[i].min;
	  else if (off2[i] > bounds[i].max)
	    off2[i] = bounds[i].max;

	}
      else			// variaveis inteiras                           
	{
	  if (flip (0.5))
	    {
	      off1[i] = p1[i];
	      off2[i] = p2[i];
	    }
	  else
	    {
	      off1[i] = p2[i];
	      off2[i] = p1[i];
	    }
	}
    }
}



void
BLX_alpha_crossover (int ind_size, double p1[], double p2[], double off1[],
		     double off2[], double blx_alpha, vect_domain bounds)
{
  double max, min, dif;
  int i;

  for (i = 0; i < ind_size; i++)
    {
      if (!isInt (bounds[i].decimals))
	{
	  max = p1[i];
	  min = p2[i];
	  if (p2[i] > max)
	    {
	      max = p2[i];
	      min = p1[i];
	    }
	  dif = max - min;
	  min = min - (dif * blx_alpha);
	  if (min < bounds[i].min)
	    min = bounds[i].min;

	  max = max + (dif * blx_alpha);
	  if (max > bounds[i].max)
	    max = bounds[i].max;

	  off1[i] = min + (uniform01 () * (max - min));
	  off2[i] = min + (uniform01 () * (max - min));
	}
      else			// variaveis inteiras                           
	{
	  if (flip (0.5))
	    {
	      off1[i] = p1[i];
	      off2[i] = p2[i];
	    }
	  else
	    {
	      off1[i] = p2[i];
	      off2[i] = p1[i];
	    }
	}
    }
}


void
apply_crossover (JOB * jo)
{
  int i;			// positions
  ind c1, c2;

  // iterates jo->new_population and produce offspring
  for (i = 0; i < jo->pop_size; i += 2)
    {
      if (flip (jo->ga.p_cx) == TRUE)
	{
	  c1.genes = malloc (sizeof (double) * jo->N);
	  c2.genes = malloc (sizeof (double) * jo->N);

	  switch (jo->ga.cx_type)
	    {
	    case 0:
	      single_point_crossover (jo->N, jo->new_population[i].genes,
				      jo->new_population[i + 1].genes,
				      c1.genes, c2.genes);
	      break;

	    case 1:
	      double_point_crossover (jo->N, jo->new_population[i].genes,
				      jo->new_population[i + 1].genes,
				      c1.genes, c2.genes);
	      break;

	    case 2:
	      BLX_alpha_crossover (jo->N, jo->new_population[i].genes,
				   jo->new_population[i + 1].genes, c1.genes,
				   c2.genes, jo->ga.blx_alpha, jo->bounds);
	      break;

	    case 3:
	      SBX_crossover (jo->N, jo->new_population[i].genes,
			     jo->new_population[i + 1].genes, c1.genes,
			     c2.genes, jo->ga.eta_sbx, jo->bounds);
	      break;

	    default:
	      puts ("Crossover type error");
	      exit (EXIT_FAILURE);
	      break;
	    }

	  free (jo->new_population[i].genes);
	  free (jo->new_population[i + 1].genes);
	  jo->new_population[i] = c1;
	  jo->new_population[i + 1] = c2;
	}
    }
}
