/*
(c) GAFit toolkit $Id: charmm.c 510 2025-04-22 14:23:13Z ro $
*/
#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <math.h>



#include "../inputline/line.h"
#include "../flyctl/flyctl.h"
#include "../environ/environ.h"
#include "../nullist/nllist.h"
#include "../rstrings/rstrings.h"
#include "../rangecf/rangecf.h"
#include "charmm.h"

int debug = 0;

char *ExternalInput = NULL;
char *ExternalFit = NULL;
char *Bounds = NULL;
char *InputTemplate = NULL;
char *CharmmParameters = NULL;
char *CharmmExecutable = NULL;
char *CharmmJobFile = NULL;
char *CharmmGeometries = NULL;
char *CharmmReferenceGeom = NULL;
char *CharmmCalculatedEnergies = NULL;



//reference
double *refenergy;
double *refwei;
int reftotal;
char **geoms;

//calculated
double *calcenergy;
int calctotal;
char **xgeoms;

char *chmcolon[] = { ":", NULL };
char *chmwhite[] = { " ", "\t", NULL };

void
InitEnvironmentVariables (void)
{
  ExternalInput = SetEnv (STR_EXTERNAL_INPUT, EXTERNAL_INPUT);
  ExternalFit = SetEnv (STR_EXTERNAL_FIT, EXTERNAL_FIT);
  Bounds = SetEnv (STR_BOUNDS_FILE, BOUNDS_FILE);
  InputTemplate = SetEnv (STR_INPUT_TEMPLATE, CHARMM_TEMPLATE);
  CharmmParameters = SetEnv (STR_CHARMM_PARAMETERS, CHARMM_PARAMETERS);
  CharmmExecutable = SetEnv (STR_CHARMM_EXECUTABLE, CHARMM_EXECUTABLE);
  CharmmJobFile = SetEnv (STR_CHARMM_JOBFILE, CHARMM_JOBFILE);
  CharmmGeometries = SetEnv (STR_CHARMM_GEOMETRIES, CHARMM_GEOMETRIES);
  CharmmReferenceGeom =
    SetEnv (STR_CHARMM_REFERENCE_GEOM, CHARMM_REFERENCE_GEOM);
  CharmmCalculatedEnergies =
    SetEnv (STR_CHARMM_CALCULATED_ENERGIES, CHARMM_CALCULATED_ENERGIES);
}


double *
addDouble (double *d, char *s, int total)
{
  double a;
  d = realloc (d, total * sizeof (double));
  if (sStr2Double (s, &a))
    {
      d[total - 1] = a;
    }
  else
    {
      d[total - 1] = 1;
    }
  return d;
}


void
extractData (char *file, char *id)
{
  FILE *r;
  char *s;
  char **list;

  r = fopen (file, "r");
  if (r)
    {
      if ((s = InputFileLine (r)))
	{
	  reftotal++;
	  list = NULL;
	  list = nllListParser (s, chmcolon);
	  refenergy = addDouble (refenergy, nllString (list, 2), reftotal);
	  refwei = addDouble (refwei, nllString (list, 4), reftotal);
	  geoms = nllAdd (geoms, sUpperCase (strdup (id)));
	  nllClear (list);
	  free (s);
	}
      else
	{
	  printf ("Cannot process %s\n", file);
	}
      fclose (r);
      return;
    }
  fcStopIt ("extact data: cannot open file");
}

void
referenceTable (char *directory, char *reference)
{
  DIR *dr;
  FILE *w;
  struct dirent *dp;
  struct stat statbuf;
  char *dir;
  char *file;
  int i, j;
  double dtmp;
  char *ctmp;
  float ereference;

  if (directory[strlen (directory)] != '/')
    dir = MakePath (directory, "/");
  else
    dir = strdup (directory);

  dr = opendir (dir);
  if (dr != NULL)
    {
      while ((dp = readdir (dr)) != NULL)
	{
	  file = MakePath (dir, dp->d_name);
	  if (stat (file, &statbuf) != 0)
	    fcStopIt ("Cannot stat file\n");
	  if (S_ISREG (statbuf.st_mode))
	    {
	      extractData (file, dp->d_name);
	    }
	  free (file);

	}
      closedir (dr);
    }
  else
    {
      printf ("Couldn't open %s\n", directory);
      fcStopIt ("Couldn't open the directory");
    }
  free (dir);

  for (i = 0; i < reftotal - 1; i++)
    {
      for (j = i + 1; j < reftotal; j++)
	{
	  if (strcmp (nllString (geoms, i), nllString (geoms, j)) > 0)
	    {
	      ctmp = geoms[i];
	      geoms[i] = geoms[j];
	      geoms[j] = ctmp;
	      dtmp = refenergy[i];
	      refenergy[i] = refenergy[j];
	      refenergy[j] = dtmp;
	      dtmp = refwei[i];
	      refwei[i] = refwei[j];
	      refwei[j] = dtmp;
	    }
	}
    }

  printf ("Energy reference:%s\n", reference);
  if ((j = nllLocate (reference, geoms)) > -1)
    {
      ereference = refenergy[j];
      if (debug)
	printf ("reference: %s, value=%f, pos=%d\n", reference, ereference,
		j);
      for (i = 0; i < reftotal; i++)
	{
	  if (debug)
	    printf ("%d %f - (%f)->", i, refenergy[i], ereference);
	  refenergy[i] = refenergy[i] - ereference;
	  if (debug)
	    printf ("%f\n", refenergy[i]);

	}
    }

  w = fopen (REFERENCE_TABLE, "w");
  if (w)
    {
      for (i = 0; i < reftotal; i++)
	{
	  fprintf (w, "%s %f %f\n", geoms[i], refenergy[i], refwei[i]);
	}
      fclose (w);
    }
  else
    fcStopIt ("Couldn't write reference table");
}

void
loadReferenceTable (void)
{
  FILE *r;
  char *s;
  char **list;

  r = fopen (REFERENCE_TABLE, "r");
  if (r)
    {
      reftotal = 0;
      while ((s = InputFileLine (r)) != NULL)
	{
	  reftotal++;
	  list = nllListParser (s, chmwhite);
	  list = nllDelete (list, chmwhite);
	  geoms = nllAdd (geoms, strdup (nllString (list, 0)));
	  refenergy = addDouble (refenergy, nllString (list, 1), reftotal);
	  refwei = addDouble (refwei, nllString (list, 2), reftotal);
	  free (s);
	  nllClear (list);
	}
      fclose (r);
      return;
    }
  fcStopIt ("load reference table: cannot open file\n");
}

void
loadCalculatedEnergies (char *reference, int icol, int ecol)
{
  FILE *r;
  char *s;
  char **list;
  int i, j;
  double dtmp;
  double calcreference;
  char *ctmp;

  if (xgeoms)
    {
      nllClear (xgeoms);
      free (calcenergy);
      xgeoms = NULL;
      calcenergy = NULL;
    }

  calctotal = 0;

  r = fopen (CharmmCalculatedEnergies, "r");
  if (r)
    {
      while ((s = InputFileLine (r)) != NULL)
	{
	  calctotal++;
	  list = nllListParser (s, chmwhite);
	  list = nllDelete (list, chmwhite);
	  xgeoms = nllAdd (xgeoms, strdup (nllString (list, icol - 1)));
	  calcenergy =
	    addDouble (calcenergy, nllString (list, ecol - 1), calctotal);
	  free (s);
	  nllClear (list);
	}
      fclose (r);

      if (calctotal != reftotal)
	fcStopIt ("the reference energies are distinct from calculated\n");

      for (i = 0; i < calctotal - 1; i++)
	{
	  for (j = i + 1; j < calctotal; j++)
	    {
	      if (strcmp (nllString (xgeoms, i), nllString (xgeoms, j)) > 0)
		{
		  ctmp = xgeoms[i];
		  xgeoms[i] = xgeoms[j];
		  xgeoms[j] = ctmp;
		  dtmp = calcenergy[i];
		  calcenergy[i] = calcenergy[j];
		  calcenergy[j] = dtmp;
		}
	    }
	}

      if (debug)
	{
	  printf ("loading calculated energies\n");
	  printf ("reference geometry:%s\n", reference);
	}

      if ((j = nllLocate (reference, xgeoms)) > -1)
	{
	  calcreference = calcenergy[j];
	  if (debug)
	    printf ("calculated reference: %s, value=%f\n", reference,
		    calcreference);
	  for (i = 0; i < calctotal; i++)
	    {
	      if (debug)
		printf ("%d %f -> ", i, calcenergy[i]);
	      calcenergy[i] -= calcreference;
	      if (debug)
		printf ("%f\n", calcenergy[i]);
	    }
	}
      else
	{
	  if (debug)
	    printf ("reference geometry %s not found\n", reference);
	}

      if (debug)
	{
	  printf (" calculated energies|| reference energies\n");
	  printf ("i ->file calculated || file reference weight\n");
	  for (i = 0; i < calctotal; i++)
	    {
	      printf ("%d ->%s %f|| %s %f %f\n", i, nllString (xgeoms, i),
		      calcenergy[i], nllString (geoms, i), refenergy[i],
		      refwei[i]);
	    }
	}

      return;
    }

  fcStopIt ("load calculated energies: cannot open file");
}

void
checkGeoms (void)
{
  int i;
  for (i = 0; i < reftotal; i++)
    {
      if (strcmp (nllString (geoms, i), nllString (xgeoms, i)) != 0)
	{
	  fcStopIt ("geometry indexes differ reference/calculated");
	}
    }
}

double
calculateFit (void)
{
  int i;
  double sum;

  sum = 0;
  for (i = 0; i < reftotal; i++)
    {
      sum += pow (calcenergy[i] - refenergy[i], 2.0) * refwei[i];
    }
  return sum;
}

void
printEvaluation (void)
{
  printf ("#\n");
  if (CharmmReferenceGeom != NULL)
    printf ("# (relative values to %s)\n", CharmmReferenceGeom);
  printf ("#\n");
  printf ("#   Geometry       Reference   Calculated   Difference\n");
  printf ("#==============   ==========  ===========  ===========\n");
  for (int i = 0; i < reftotal; i++)
    {
      printf ("  %-13s %12f %12f %11.4f%%\n", xgeoms[i], refenergy[i],
	      calcenergy[i],
	      refenergy[i] ==
	      0.0 ? 0.0 : (refenergy[i] -
			   calcenergy[i]) / refenergy[i] * 100);
    }
// to debug:    printf("%g\n",calculateFit());
}

int
WriteFile (char *content, char *file)
{
  FILE *w;
  w = fopen (file, "w");
  if (w)
    {
      fprintf (w, "%s", content);
      fclose (w);
      return 1;
    }
  fcStopIt ("cannot write file\n");
  return 0;
}


void
chmmClean (RANGECOEFS * rc)
{
  rcTemplateClean (rc);
  nllClear (geoms);
  nllClear (xgeoms);
  free (refwei);
  free (refenergy);
  free (calcenergy);
}

void
writeFinalHint (int i, int e, char *einput, char *efit, char *cmd,
		char *template, char *parameters, char *executable, char *job,
		char *rgeom, char *cener)
{
  FILE *w;
  w = fopen (FINAL_HINT, "w");
  if (w)
    {
      fprintf (w, "%d %d\n", i, e);
      fprintf (w, "%s\n", einput);
      fprintf (w, "%s\n", efit);
      fprintf (w, "%s\n", cmd);
      fprintf (w, "%s\n", template);
      fprintf (w, "%s\n", parameters);
      fprintf (w, "%s\n", executable);
      fprintf (w, "%s\n", job);
      fprintf (w, "%s\n", rgeom);
      fprintf (w, "%s\n", cener);
      fclose (w);
      return;
    }
  fcStopIt ("cannot write chmfinal-hint");
}

void
readFinalHint (int *i, int *e, char *einput, char *efit, char *cmd,
	       char *template, char *parameters, char *executable, char *job,
	       char *rgeom, char *cener)
{
  FILE *r;
  r = fopen (FINAL_HINT, "r");
  if (r)
    {
      fscanf (r, "%d", i);
      fscanf (r, "%d", e);;
      fscanf (r, "%s", einput);
      fscanf (r, "%s", efit);
      fscanf (r, "%s", cmd);
      fscanf (r, "%s", template);
      fscanf (r, "%s", parameters);
      fscanf (r, "%s", executable);
      fscanf (r, "%s", job);
      fscanf (r, "%s", rgeom);
      fscanf (r, "%s", cener);
      fclose (r);

      setenv (STR_EXTERNAL_INPUT, einput, 0);
      setenv (STR_EXTERNAL_FIT, efit, 0);
      setenv (STR_INPUT_TEMPLATE, template, 0);
      setenv (STR_CHARMM_PARAMETERS, parameters, 0);
      setenv (STR_CHARMM_EXECUTABLE, executable, 0);
      setenv (STR_CHARMM_JOBFILE, job, 0);
      setenv (STR_CHARMM_PARAMETERS, parameters, 0);
      setenv (STR_CHARMM_REFERENCE_GEOM, rgeom, 0);
      setenv (STR_CHARMM_CALCULATED_ENERGIES, cener, 0);

      return;
    }
  fcStopIt ("cannot read chmfinal - hint ");
}
