/*
(c) GAFit toolkit $Id: analytical.c 181 2016-07-12 10:21:58Z ro $
*/
#if HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "../fpu/fpu.h"
#include "../nullist/nllist.h"
#include "../compiler/ucompiler.h"
#include "../parameters/parameters.h"
#include "../inputline/line.h"
#include "analytical.h"

char *tlist[] = { ",", " ", "\t", NULL };

static FPU2011 *fpu;

static char **vars = NULL;
static int nvars, npot, ndist, ncoefs;
static char **lcoefs = NULL;
static int *icoefs = NULL;


char *
anPotVar (void)
{
  return nllString (vars, npot);
}

char *
anDistVar (void)
{
  return nllString (vars, ndist);
}

char *
anCoefVar (int i)
{
  return nllString (vars, icoefs[i]);
}

int
anGetNCoefs (void)
{
  return ncoefs;
}

char **
tokenSep (void)
{
  return tlist;
}

int
anCoeffIndex (int ncoefs, int **icoefs, char **lcoefs, char **vars)
{
  int isOk = 1;
  int i;
  int *ic = (int *) malloc (sizeof (int) * ncoefs);

  for (i = 0; i < ncoefs; i++)
    {
      ic[i] = nllLocate (*(lcoefs + i), vars);
      if (ic[i] == -1)
	{
	  printf ("coefficient %s no in expression\n", *(lcoefs + i));
	  isOk = 0;
	}
#ifdef __DEBUG__
      printf ("coefficient %s index %d\n", *(lcoefs + i), ic[i]);
#endif
    }
  if (isOk)
    {
      *icoefs = ic;
      return 1;
    }
  free (ic);
  return 0;
}

int
anGetParameters (char *job, char *potname, char *potvar, char *potdist,
		 char *potcoef, int maxstr)
{
  return GetStringParameter
    (job, "analytical", "expression", "STOP", potname, maxstr)
    && GetStringParameter (job, "analytical", "potential", "STOP",
			   potvar, maxstr)
    && GetStringParameter (job, "analytical", "distance", "STOP",
			   potdist, maxstr)
    && GetStringParameter (job, "analytical", "coefficients", "STOP",
			   potcoef, maxstr);
}

void
anPrintPrettyExpr (char *l)
{
  char *p = l;
  printf ("\n\t");
  do
    {
      if (*p == ';')
	printf (";\n\t");
      else if (*p == '\n');
      else if (*p == '=')
	printf (" = ");
      else
	printf ("%c", *p);
    }
  while (*p++);
  printf ("\n\n");
}

void
anCheckAnalytical (char *job, char *file, int print)
{
  char **result;
  char potname[MAXSTR], potvar[MAXSTR], potdist[MAXSTR], potcoef[MAXSTR];
  char *pfile;

  char *l;

  pfile = MakePath (file, ".uxe");

  if (anGetParameters (job, potname, potvar, potdist, potcoef, MAXSTR))
    {
      if (print)
	{
	  printf ("expression name: \"%s\"\n", potname);
	  printf ("potential:       %s\n", potvar);
	  printf ("distance:        %s\n", potdist);
	  printf ("coefficients:    %s\n", potcoef);
	  printf ("\n");
	}
      l = SectionText (job, potname);
      if (l)
	{
	  if (print)
	    {
	      printf ("Expression found:\n");
	      anPrintPrettyExpr (l);
	    }
	  result = uCompile (l, file, &vars);
	  free (l);
	  if (result)
	    {
	      nllClear (result);
	      nvars = nllCount (vars);
	      npot = nllLocate (potvar, vars);
	      ndist = nllLocate (potdist, vars);
	      lcoefs = nllParser (potcoef, tokenSep);
	      lcoefs = nllDelete (lcoefs, tlist);
	      ncoefs = nllCount (lcoefs);
	      if (print)
		{
		  nllPrint (vars, "\tVariables found in expression: ");
		  printf ("\n");
		  printf ("\tExpression code OK\n");
		  printf ("\t%s index %d\n", potvar, npot);
		  printf ("\t%s index %d\n", potdist, ndist);
		  printf ("\t%d coefficients found\n", ncoefs);
		}
	      if (npot != -1 && ndist != -1)
		{
		  if (anCoeffIndex (ncoefs, &icoefs, lcoefs, vars))
		    {
		      if (FileExist (pfile))
			{
			  fpu = fpuNew ();
			  fpuLoad (fpu, pfile);
			  free (pfile);
			  return;
//************************************RETURN HERE*********
//**************************FPU LOADED AND READY TO RUN***
			}
		      else
			printf ("No prog file produced\n");
		    }
		  else
		    printf ("No coefficients in expression\n");
		}
	      else
		printf ("Potential or distance no present in expression\n");
	    }
	  else
	    printf ("No code produced\n");
	}
      else
	printf ("No section with potential expression\n");
    }
  else
    printf ("No expression\n");
  exit (EXIT_FAILURE);
}

//ufpu calls come here (see Analytical function)
int
anSetAndRun (double *potential, double distance, double *coefs, int print)
{
  int i;
  int flags;
  double *mblock = (double *) malloc (sizeof (double) * nvars);
  //note that we don't know a priori where are the variables layered
  //from the expresion compiled code... if expression changes 
  //then layout changes
  mblock[npot] = *potential;
  mblock[ndist] = distance;
  for (i = 0; i < ncoefs; i++)
    mblock[icoefs[i]] = coefs[i];

  //passing entire block in correct order as memory for fpu 
  fpuMemoryBlockAdd (fpu, mblock, nvars);
  //also can be done by fpuAddVariable, in this case, variables
  //must be added in correct order and are passed by reference
  //ie: fpu works with the real variables and it changes them in site.

  //now we run fpu
  flags = fpu2011 (fpu);
  if (print)
    {
      printf ("\nAfter run:");
      fpuPrintMemoryLabels (fpu, vars, nvars);
    }
  //get potential value after calculation
  *potential = fpuGetVariable (fpu, npot);
  //fpuReset doesn't freed this block, is protected by "MemoryProtect".
  free (mblock);
  fpuReset (fpu);
//fpu ready to run again.  
  return flags;
}

//clears all. No more fpu to run.
void
anClearAll (void)
{
  fpuClear (fpu);
  free (fpu);
  nllClear (vars);
  nllClear (lcoefs);
  free (icoefs);
}

//hook to fortran code. potRouter (potential.f) calls
//come here.
//index begins in 1 (call from fortran)
//so for C code, index=index-1
double
Analytical (double *distance, int *index, double *coefs)
{
  double potential = 0;
  int status;
#ifdef __DEBUG__
  int i;
  printf ("distance: %lf\n", *distance);
  printf ("index:%d\n", *index);
  for (i = 0; i < ncoefs; i++)
    printf ("coef %d = %lf\n", i, *(coefs + (*index - 1) + i));
#endif
  if ((status =
       anSetAndRun (&potential, *distance, (double *) coefs + (*index - 1),
		    0)))
    {
      fpuExplainStatus (status);
      exit (EXIT_FAILURE);
    }
  return potential;
}
